use bevy::prelude::*;
use http::Uri;

#[cfg(feature = "graphics")]
use crate::game::{bot::data::TileClickedEvent, controls::placer::TileClickEvent};
use crate::{
    coordinates::CubeCoordinate,
    game::{
        bot::{
            data::{BotEvent, TileStatusEvent, TileStatusKind},
            BotConnecting, BotConnectingState, BotSettings, BotState, ConnectingTimeout,
        },
        gamemodes::creative::CreativeGameState,
        map::{Map, TileSaveData},
        resources::{MarketConfiguredEvent, TilePlacedEvent, TileReclaimedEvent},
        tiles::{
            chameleon_tile::ChameleonTileChangedEvent, rgb_tile::RgbTileChangedEvent,
            ConsistentTileData, TileType,
        },
        DisplayMode, GameConfig,
    },
    prelude::*,
    websocket::WebSocketConnector,
};

macro_rules! define_creative_bot_plugin {
    ($({
        // gamemode to handle
        gamemode: $gm:expr,
        // state in which creative bot should be generated
        state: $state:expr,
    }),*) => {
        pub struct CreativeBotPlugin{
            pub mode: DisplayMode,
        }
        impl Plugin for CreativeBotPlugin {
            fn build(&self, app: &mut App) {
                #[cfg(feature = "graphics")]
                if let DisplayMode::Graphic = self.mode {
                    app.add_systems(
                        Update,
                        tile_click_event
                            .run_if(on_event::<TileClickEvent>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                }
                app.add_event::<TileStatusReq>();
                app.add_systems(
                    Update,
                    tile_status_event.run_if(never()$(.or(still_in_state($state)))*)
                );
            }
        }
    };
}

define_creative_bot_plugin![
    {
        gamemode: Creative,
        state: CreativeGameState::InGame,
    }
];

pub fn start_creative_bot_game(
    mut commands: Commands,
    mut settings: ResMut<BotSettings>,
    mut bot_state: ResMut<BotState>,
    mut state: ResMut<BotConnectingState>,
    config: Res<GameConfig>,
    mut timer: ResMut<ConnectingTimeout>,
) {
    let Ok(uri) = Uri::builder()
        .scheme("ws")
        .authority(config.bot_url.clone())
        .path_and_query("/")
        .build()
    else {
        warn!("Parsing Bot url failed.");
        *state = BotConnectingState::UrlError;
        return;
    };

    let Ok(connector) = WebSocketConnector::new(uri) else {
        *state = BotConnectingState::UrlError;
        warn!("Calling Bot url failed.");
        return;
    };

    commands.insert_resource(BotConnecting(connector));

    *state = BotConnectingState::WaitingWs;
    settings.action_limit = None;
    settings.disable_on_action = false;
    bot_state.execute_actions = true;
    bot_state.registered_events.clear();
    timer.reset();
}

#[cfg(feature = "graphics")]
pub fn tile_click_event(
    mut bot_event: EventWriter<BotEvent>,
    mut tile_click_event: EventReader<TileClickEvent>,
    map: Res<Map>,
    tiles: Query<&TileType>,
) {
    for event in tile_click_event.read() {
        bot_event.send(BotEvent::TileClicked(TileClickedEvent::new(
            event.tile_pos,
            map.get(event.tile_pos)
                .and_then(|id| tiles.get(id).ok())
                .cloned(),
        )));
    }
}

#[derive(Debug, Event)]
pub struct TileStatusReq(pub CubeCoordinate);

#[allow(clippy::too_many_arguments)]
pub fn tile_status_event(
    mut bot_event: EventWriter<BotEvent>,
    mut added: EventReader<TilePlacedEvent>,
    mut removed: EventReader<TileReclaimedEvent>,
    mut market: EventReader<MarketConfiguredEvent>,
    mut rgb: EventReader<RgbTileChangedEvent>,
    mut chameleon: EventReader<ChameleonTileChangedEvent>,
    mut tile_stat: EventReader<TileStatusReq>,

    tiles: Query<ConsistentTileData, With<TileType>>,
    map: Res<Map>,
) {
    for add in added.read() {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            add.0.coord,
            TileStatusKind::Added,
            Some(add.0.clone()),
        )));
    }

    for remove in removed.read() {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            remove.0,
            TileStatusKind::Removed,
            None,
        )));
    }

    for (coord, kind) in market
        .read()
        .map(|market| (market.coord, TileStatusKind::Changed))
        .chain(rgb.read().map(|evt| (evt.0, TileStatusKind::Changed)))
        .chain(chameleon.read().map(|evt| (evt.0, TileStatusKind::Changed)))
        .chain(tile_stat.read().map(|evt| (evt.0, TileStatusKind::Manual)))
    {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            coord,
            kind,
            map.get(coord)
                .and_then(|id| tiles.get(id).ok())
                .map(|data| TileSaveData {
                    coord: *data.0,
                    tile: *data.1,
                    data: TileType::serialize_tile_data(data),
                    reclaim: data.2.map(|f| f.timer.remaining_secs()),
                }),
        )));
    }
}
