use std::time::Duration;

use bevy::prelude::*;
use http::Uri;

use crate::{
    coordinates::CubeCoordinate,
    game::{
        build_area::{setup_build_area, BuildArea},
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
        },
        hand::{generate_hand, reset_deck, Hand},
        map::RawMap,
        metrics::{reset_metrics, reset_target_metrics, Metrics, MetricsRate, TargetMetrics},
        reset_game_state,
        resources::{ResourceTickEvent, RESOURCE_TICK_INTERVAL},
        rewards::{reset_rewards_timer, Reward, RewardSend},
        round::{
            continue_timer, reset_round_resources, update_redraw_cost, HandRedrawCost,
            NewHandTimer, RoundCounter, RoundTimer,
        },
        tiles::{ConsistentTileData, TileType},
        time::{reset_game_time, unpause_game_time, StepEvent},
        GameSimSet,
    },
    prelude::still_in_state,
    websocket::WebSocketConnector,
    GameConfig,
};

use super::{
    data::{BotEvent, BotEventKind, ChallengeTick},
    in_game_handler, ActionEvent, BotConnecting, BotConnectingState, BotConnection, BotSettings,
    BotState, BotTimeoutEvent, ConnectingTimeout,
};

const BOT_RESPONSE_TIMEOUT: f32 = 0.5;

macro_rules! define_challenge_bot_plugin {
    ($({
        // gamemode to handle
        gamemode: $gm:expr,
        // state in which challenge bot should be generated
        state: $state:expr,
    }),*) => {
        pub struct ChallengeBotPlugin;
        impl Plugin for ChallengeBotPlugin {
            fn build(&self, app: &mut App) {
                use crate::prelude::never;
                app.add_systems(
                    Update,
                    simulate_bot
                        .run_if(never()$(.or(still_in_state($state)))*)
                        .run_if(resource_exists::<BotConnection>)
                        .run_if(on_event::<ResourceTickEvent>)
                        .in_set(GameSimSet::NextAuto),
                );
                $(
                    app.add_systems(
                        OnEnter($state),
                        simulate_bot
                            .after(continue_timer)
                            .after(update_redraw_cost)
                            .after(reset_round_resources)
                            .after(generate_hand)
                            .after(reset_deck)
                            .after(reset_game_state)
                            .after(reset_game_time)
                            .after(reset_metrics)
                            .after(unpause_game_time)
                            .after(setup_build_area)
                            .after(reset_target_metrics)
                            .after(reset_rewards_timer)
                            .run_if(resource_exists::<BotConnection>),
                    );
                )*
                app.add_systems(
                    Update,
                    tick_headless_time
                        .after(in_game_handler)
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(resource_exists::<BotConnection>),
                );
                app.init_resource::<WaitForTickResponse>();
            }
        }
    }
}

define_challenge_bot_plugin![
    {
        gamemode: GameMode::Challenge,
        state: ChallengeGameState::InRound,
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        state: CampaignChallengeState::InRound,
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        state: CampaignWettbewerbState::InRound,
    },
    {
        gamemode: GameMode::Zen,
        state: ZenGameState::InRound,
    }
];

pub fn start_challenge_bot_game(
    mut commands: Commands,
    mut settings: ResMut<BotSettings>,
    mut bot_state: ResMut<BotState>,
    mut state: ResMut<BotConnectingState>,
    config: Res<GameConfig>,
    mut timer: ResMut<ConnectingTimeout>,
    mut wait: ResMut<WaitForTickResponse>,
) {
    let Ok(uri) = Uri::builder()
        .scheme("ws")
        .authority(config.bot_url.clone())
        .path_and_query("/")
        .build()
    else {
        warn!("Parsing Bot url failed.");
        *state = BotConnectingState::UrlError;
        return;
    };

    let Ok(connector) = WebSocketConnector::new(uri) else {
        *state = BotConnectingState::UrlError;
        warn!("Calling Bot url failed.");
        return;
    };

    commands.insert_resource(BotConnecting(connector));

    *state = BotConnectingState::WaitingWs;
    settings.action_limit = Some(1);
    settings.disable_on_action = true;
    bot_state.execute_actions = true;
    bot_state.registered_events.clear();
    // enable ChallengeTickEvent by default
    bot_state
        .registered_events
        .insert(BotEventKind::ChallengeTick);
    timer.reset();
    wait.reset();
    wait.unpause();
}

/// tracks first action after a challenge tick and the response timeout.
#[derive(Debug, Resource, Deref, DerefMut)]
pub struct WaitForTickResponse(pub Timer);

impl Default for WaitForTickResponse {
    fn default() -> Self {
        Self(Timer::from_seconds(BOT_RESPONSE_TIMEOUT, TimerMode::Once))
    }
}

#[allow(clippy::too_many_arguments)]
pub fn simulate_bot(
    mut bot_event: EventWriter<BotEvent>,
    tiles: Query<ConsistentTileData, With<TileType>>,
    round_counter: Res<RoundCounter>,
    round_timer: Res<RoundTimer>,
    hand: Res<Hand>,
    redraw: Res<HandRedrawCost>,
    build_area: Query<&BuildArea>,
    rewards: Query<(&Reward, &CubeCoordinate)>,
    redraw_timer: Res<NewHandTimer>,
    settings: Res<BotSettings>,
    mut state: ResMut<BotState>,
    metrics: Res<Metrics>,
    metrics_rate: Res<MetricsRate>,
    metrics_target: Res<TargetMetrics>,
    mut wait: ResMut<WaitForTickResponse>,
) {
    bot_event.send(BotEvent::ChallengeTick(ChallengeTick {
        map: RawMap::save_game(tiles),
        round: **round_counter,
        round_time: round_timer.remaining_secs(),
        hand: hand.items.clone(),
        build_area: build_area.get_single().cloned().unwrap_or_default(),
        redraw: redraw.clone(),
        redraw_time: redraw_timer.remaining_secs(),
        actions_count: settings.action_limit,
        rewards: rewards
            .iter()
            .map(|(reward, coord)| RewardSend {
                reward: *reward,
                coord: *coord,
            })
            .collect::<_>(),
        metrics: metrics.clone(),
        metrics_rate: metrics_rate.clone(),
        metrics_target: metrics_target.clone(),
    }));
    state.execute_actions = true;
    wait.reset();
    wait.unpause();
}

pub fn tick_headless_time(
    mut time_stepper: EventWriter<StepEvent>,
    mut action: EventReader<ActionEvent>,
    mut wait: ResMut<WaitForTickResponse>,
    time: Res<Time<Real>>,
    mut state: ResMut<BotState>,
    mut timeout: EventWriter<BotTimeoutEvent>,
) {
    wait.tick(time.delta().min(Duration::from_millis(100)));
    if wait.just_finished() {
        time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
        wait.pause();
        state.execute_actions = false;
        timeout.send(BotTimeoutEvent);
        warn!("Bot missed action timeout.");
    } else if !action.is_empty() {
        // received events includes at least one restricted event, indicating a response to the challengeTickEvent.
        let restricted = action
            .read()
            .fold(false, |a, b| a || b.restricted || b.empty);
        if !restricted {
            return;
        }
        action.clear();
        if !wait.finished() && !wait.paused() {
            time_stepper.send(StepEvent(RESOURCE_TICK_INTERVAL));
            wait.pause();
        }
    }
}
