use std::{marker::PhantomData, time::Duration};

use bevy::prelude::*;

use crate::{errors::ErrorDisplay, i18n::Localization, AppState, DisplayMode};

#[derive(Default)]
pub struct AssetLoadStateDump {
    /// home many assets are required to load
    pub requested: usize,
    /// how many assets were loaded
    pub loaded: usize,
    /// number of assets that failed to load
    pub failed: usize,
    /// number of assets that haven't been loaded yet,
    /// or failed, but we don't care if they do (optional assets)
    pub ignored: usize,
}

pub trait LoadableAssetCollection: Resource + Sized {
    /// initializes the resource
    /// loads all required assets directly into a handle,
    /// optional assets are wrappen in Options
    fn load_all(asset_server: &AssetServer) -> Self;
    /// checks if all handles were loaded
    /// if optional handles failed to load, they will be reset to None
    fn check_all(&mut self, asset_server: &AssetServer) -> AssetLoadStateDump;

    /// return the error display configuration
    /// return None to ignore the error
    #[allow(unused_variables)]
    fn get_error(&self, localization: Option<&Localization>) -> Option<ErrorDisplay> {
        None
    }

    /// registers all required components to load the collection
    /// you probably want to use `LACManager.register_asset_collection` instead
    fn register(app: &mut App) -> &mut App {
        app.add_systems(Startup, register_and_load_assets::<Self>);
        app.add_event::<AssetLoadedEvent<Self>>();
        app.add_systems(
            Update,
            check_asset_load_state::<Self>
                .run_if(in_state(AppState::LoadApp))
                .before(wait_for_assets)
                .after(reset_asset_tracker),
        );
        app
    }
}

#[derive(Event)]
pub struct AssetLoadedEvent<T: LoadableAssetCollection>(pub PhantomData<T>);

#[macro_export]
/// Easily define LoadableAssetCollections
/// the name determines the name of the resource which contains all the handles
/// ```ignore
/// use terratactician_expandoria::define_asset_collection;
/// use bevy::prelude::*;
/// define_asset_collection!(
///     Demo,
///     !my_required_asset : Image = "images/demo.png",
///     ?my_optional_asset : Image = "images/demo2.png",
///     );
/// ```
/// When required assets fail to load, an error screen will be shown
/// When optional assets fail, everything just moves on normally
macro_rules! define_asset_collection {
    ($name:ident, $(!$raa:ident : $rat:ty = $rap:literal,)* $(?$oaa:ident : $oat:ty = $oap:literal,)* $(err : $title_id:literal $title:literal $desc_id:literal $desc:literal)?) => {
        #[derive(Default, Resource)]
        pub struct $name {
            $(
                /// loaded from
                #[doc = $rap]
                /// guaranteed to be loaded after the AppLoad state
                pub $raa: Handle<$rat>,
            )*
            $(
                /// loaded from
                #[doc = $oap]
                /// if the asset loaded successfully, the option is set to Some(Handle)
                /// if the asset failed to load, the option has the variant None
                /// During the AppLoad phase the option will always be set to the Some variant
                pub $oaa: Option<Handle<$oat>>,
            )*
        }

        impl $crate::game::asset_loading::LoadableAssetCollection for $name {
            fn load_all(asset_server: &bevy::asset::AssetServer) -> Self {
                Self {
                    $($raa: asset_server.load($rap),)*
                    $($oaa: Some(asset_server.load($oap)),)*
                }
            }
            fn check_all(&mut self, asset_server: &bevy::asset::AssetServer) -> $crate::game::asset_loading::AssetLoadStateDump {
                let mut dump = $crate::game::asset_loading::AssetLoadStateDump::default();

                // check required assets
                $(
                    dump.requested += 1;
                    match asset_server.get_load_state(&self.$raa) {
                        Some(bevy::asset::LoadState::Loaded) => dump.loaded += 1,
                        Some(bevy::asset::LoadState::Failed(_)) => dump.failed += 1,
                        _ => ()
                    }
                )*
                // check optional assets
                $(
                    dump.requested += 1;
                    if let Some(handle) = &self.$oaa {
                        match asset_server.get_load_state(handle) {
                            Some(bevy::asset::LoadState::Loaded) => dump.loaded += 1,
                            Some(bevy::asset::LoadState::Failed(_)) => {
                                dump.ignored += 1;
                                // mark asset as unavailable
                                self.$oaa = None;
                            },
                            _ => ()
                        }
                    } else {
                        // optional asset failed to load in an earlier frame
                        dump.ignored += 1;
                    }
                )*

                dump
            }

            $(
                fn get_error(&self, localization: Option<&$crate::i18n::Localization>) -> Option<$crate::errors::ErrorDisplay> {
                    use $crate::i18n::Translate;
                    Some($crate::errors::ErrorDisplay {
                        title: localization
                            .and_then(|l| l.try_translate($title_id))
                            .unwrap_or(String::from($title)),
                        description: localization
                            .and_then(|l| l.try_translate($desc_id))
                            .unwrap_or(String::from($desc)),
                        link: None,
                    })
                }
            )?
        }
    };
}

#[derive(Default, Resource)]
pub struct AppLoadingState {
    /// amount of assets we are trying to load
    pub requested: usize,
    /// amount of assets we have already loaded
    pub loaded: usize,
    /// number of assets that failed to load
    pub failed: usize,
    /// number of assets that haven't been loaded yet,
    /// or failed, but we don't care if they do (optional assets)
    pub ignored: usize,
}

pub fn register_and_load_assets<T>(
    mut state: ResMut<AppLoadingState>,
    asset_server: Res<AssetServer>,
    mut commands: Commands,
) where
    T: LoadableAssetCollection,
{
    let asset_server = asset_server.into_inner();
    let mut collection = T::load_all(asset_server);
    let dump = collection.check_all(asset_server);
    commands.insert_resource(collection);
    state.requested += dump.requested;
}

#[allow(clippy::too_many_arguments)]
fn check_asset_load_state<T>(
    mut collection: ResMut<T>,
    mut state: ResMut<AppLoadingState>,
    asset_server: Res<AssetServer>,
    mut next: ResMut<NextState<AppState>>,
    mut commands: Commands,
    localization: Option<Res<Localization>>,
    mut event: EventWriter<AssetLoadedEvent<T>>,
    mut finished_state: Local<Option<AssetLoadStateDump>>,
) where
    T: LoadableAssetCollection,
{
    // the asset finished loading
    if let Some(dump) = &*finished_state {
        state.loaded += dump.loaded;
        state.failed += dump.failed;
        state.ignored += dump.ignored;
        return;
    }

    let dump = collection.check_all(asset_server.into_inner());

    if dump.failed > 0 {
        // show the error screen if the collection has an error message
        let msg = collection
            .get_error(localization.map(|l| l.into_inner()))
            .unwrap_or_default();
        commands.spawn(msg);

        next.set(AppState::Error);
    }

    state.loaded += dump.loaded;
    state.failed += dump.failed;
    state.ignored += dump.ignored;

    if dump.requested == dump.loaded + dump.ignored {
        // mark the collection as loaded
        *finished_state = Some(dump);
        // notifiy possible listeners
        event.send(AssetLoadedEvent(PhantomData {}));
    }
}
pub fn wait_for_assets(
    state: Res<AppLoadingState>,
    mut next_state: ResMut<NextState<AppState>>,
    delay: Res<AppLoadingDelay>,
    display_mode: Res<State<DisplayMode>>,
) {
    debug!(
        "Loaded {} assets; {} failed, {} ignored",
        state.loaded, state.failed, state.ignored
    );

    if state.requested == state.loaded + state.ignored {
        // exit app loaded state
        // only exit the loading screen if the minimal delay has elapsed
        if delay.finished() || *display_mode.get() == DisplayMode::Headless {
            next_state.set(AppState::AppMenu);
        }
    }
}
pub fn reset_asset_tracker(mut state: ResMut<AppLoadingState>) {
    // reset tracker for next frame
    state.loaded = 0;
    state.failed = 0;
    state.ignored = 0;
}

/// minimal delay until the loading screen is closed
#[cfg(not(debug_assertions))]
pub const APP_START_MIN_DELAY: Duration = Duration::from_millis(2000);
#[cfg(debug_assertions)]
pub const APP_START_MIN_DELAY: Duration = Duration::from_millis(0);

#[derive(Resource, Deref, DerefMut)]
pub struct AppLoadingDelay(pub Timer);
impl Default for AppLoadingDelay {
    fn default() -> Self {
        Self(Timer::new(APP_START_MIN_DELAY, TimerMode::Once))
    }
}
fn step_load_delay(time: Res<Time>, mut delay: ResMut<AppLoadingDelay>) {
    delay.tick(time.delta());
}

pub struct AppLoadingPlugin;
impl Plugin for AppLoadingPlugin {
    fn build(&self, app: &mut App) {
        app.init_resource::<AppLoadingDelay>();
        app.init_resource::<AppLoadingState>();
        app.add_systems(
            Update,
            (reset_asset_tracker, wait_for_assets)
                .chain()
                .run_if(in_state(AppState::LoadApp)),
        );
        app.add_systems(Update, step_load_delay.run_if(in_state(AppState::LoadApp)));
    }
}

pub trait LACManager {
    fn register_asset_collection<T>(&mut self) -> &mut Self
    where
        T: LoadableAssetCollection;
}
impl LACManager for App {
    fn register_asset_collection<T>(&mut self) -> &mut Self
    where
        T: LoadableAssetCollection,
    {
        T::register(self);
        self
    }
}
