use bevy::{
    ecs::component::Component,
    math::{Ray3d, Vec2},
    prelude::Vec3,
};
use serde::{Deserialize, Serialize};
use std::ops::{Add, AddAssign, Mul, Sub};

pub const SQRT_3: f64 = 1.732_050_807_568_877_2_f64;
pub const EDGE_LENGTH: f64 = 0.577_350_269_189_625_7_f64; // 1.0 / SQRT_3
pub const TILE_SIZE: f64 = 1.0; // Distance between tow parallel sides.

#[derive(Clone, Copy, Debug, Hash, PartialEq, Eq)]
pub struct OffsetCoordinate {
    // Hexagon Offset Coordinates
    pub col: isize,
    pub row: isize,
}

#[allow(dead_code)]
impl OffsetCoordinate {
    pub fn new(col: isize, row: isize) -> Self {
        Self { col, row }
    }
    pub fn convert_to_cube(&self) -> CubeCoordinate {
        let q = self.col - (self.row - (self.row & 1)) / 2;
        let r = self.row;
        CubeCoordinate { q, r, s: -q - r }
    }
}

impl From<OffsetCoordinate> for Vec3 {
    fn from(value: OffsetCoordinate) -> Self {
        Vec3 {
            x: value.col as f32 // every other column is offset by 0.5
                + (if value.row.rem_euclid(2) == 0 {
                    0.0
                } else {
                    0.5
                }),
            y: 0.,
            z: (value.row as f64 * (EDGE_LENGTH * 1.5)) as f32, // 0.5774 is edge length. The offset is 1.5 times this value.
        }
    }
}

fn direction_from_index(dir: usize) -> CubeCoordinate {
    match dir {
        0 => CubeCoordinate { q: 0, r: 1, s: -1 },
        1 => CubeCoordinate { q: -1, r: 1, s: 0 },
        2 => CubeCoordinate { q: -1, r: 0, s: 1 },
        3 => CubeCoordinate { q: 0, r: -1, s: 1 },
        4 => CubeCoordinate { q: 1, r: -1, s: 0 },
        5 => CubeCoordinate { q: 1, r: 0, s: -1 },
        _ => panic!("There are only 6 Directions."),
    }
}

pub struct CloseNeighborhoodIterator {
    coordinate: CubeCoordinate,
    index: usize,
}

impl Iterator for CloseNeighborhoodIterator {
    type Item = CubeCoordinate;

    fn next(&mut self) -> Option<Self::Item> {
        if self.index >= 6 {
            return None;
        }
        let coord = self.coordinate + direction_from_index(self.index);
        self.index += 1;
        Some(coord)
    }
}

pub struct NeighborhoodIterator {
    center: CubeCoordinate,
    radius: usize,
    iteration: usize,
    current_ring: RingIterator,
}

impl Iterator for NeighborhoodIterator {
    type Item = CubeCoordinate;

    fn next(&mut self) -> Option<Self::Item> {
        let next = self.current_ring.next();
        if next.is_some() {
            return next;
        }
        self.iteration += 1;
        if self.iteration > self.radius {
            return None;
        }
        self.current_ring = RingIterator::new(self.center, self.iteration);
        self.current_ring.next()
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        // Size of Neighborhood with radius r equal 3 * r * (r+1)
        // Since we want to subtract the part we have done
        // The amount left (where k rows are done) is
        //   (3 * r * (r+1)) - (3 * k * (k+1))
        // = 3 * (r + k + 1) (r - k)
        let area_left = 3 * (self.radius + self.iteration) * (self.radius - self.iteration + 1);
        let ring_part = 6 * self.current_ring.current_direction + self.current_ring.index;
        let size_hint = area_left - ring_part;
        (size_hint, Some(size_hint))
    }
}

impl NeighborhoodIterator {
    pub fn new(center: impl Into<CubeCoordinate>, radius: usize) -> Self {
        if radius == 0 {
            panic!("No neighborhood with radius 0.")
        }

        let center = center.into();

        NeighborhoodIterator {
            center,
            radius,
            iteration: 1,
            current_ring: RingIterator::new(center, 1),
        }
    }
}

pub struct RingIterator {
    current: CubeCoordinate,
    radius: usize,
    current_direction: usize,
    index: usize,
}

impl RingIterator {
    pub fn new(center: impl Into<CubeCoordinate>, radius: usize) -> Self {
        RingIterator {
            current: center.into()
                + CubeCoordinate {
                    q: radius as isize,
                    r: -(radius as isize),
                    s: 0,
                },
            radius,
            current_direction: 0,
            index: 0,
        }
    }
}

impl Iterator for RingIterator {
    type Item = CubeCoordinate;

    fn next(&mut self) -> Option<Self::Item> {
        if self.radius == 0 {
            if self.index == 0 {
                self.index += 1;
                return Some(self.current);
            } else {
                return None;
            }
        }
        if self.current_direction >= 6 {
            return None;
        }
        let ret = self.current;
        self.current += direction_from_index(self.current_direction);

        self.index += 1;
        if self.index >= self.radius {
            self.index = 0;
            self.current_direction += 1;
        }
        Some(ret)
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        if self.radius == 0 {
            return (1, Some(1));
        }
        let sides_completed = self.current_direction;
        let length_left = (6 - sides_completed) * self.radius;
        let size_hint = length_left - self.index;
        (size_hint, Some(size_hint))
    }
}

#[derive(
    Component,
    Clone,
    Copy,
    Default,
    Debug,
    Hash,
    PartialEq,
    Eq,
    PartialOrd,
    Ord,
    Serialize,
    Deserialize,
)]
/// CubeCoordinate: coordinate in hexagon world
/// condition: q+r+s = 0
/// CubeCoordinate implements Ord. Ord is a lexicographic order, but has no meaning apart from this.
pub struct CubeCoordinate {
    pub q: isize,
    pub r: isize,
    pub s: isize,
}

#[allow(dead_code)]
impl CubeCoordinate {
    pub fn new(q: isize, r: isize, s: isize) -> Self {
        Self { q, r, s }
    }

    pub fn new_rounded(q: f32, r: f32, s: f32) -> Self {
        let mut q_round = q.round();
        let mut r_round = r.round();
        let mut s_round = s.round();
        let q_diff = (q_round - q).abs();
        let r_diff = (r_round - r).abs();
        let s_diff = (s_round - s).abs();
        if q_diff > r_diff && q_diff > s_diff {
            q_round = -r_round - s_round;
        } else if r_diff > s_diff {
            r_round = -q_round - s_round;
        } else {
            s_round = -q_round - r_round;
        }
        Self {
            q: q_round as isize,
            r: r_round as isize,
            s: s_round as isize,
        }
    }

    const Q_VEC: Vec3 = Vec3 {
        x: (SQRT_3 / 2.0) as f32,
        y: 0.0,
        z: -0.5,
    };
    const R_VEC: Vec3 = Vec3 {
        x: 0.0,
        y: 0.0,
        z: 1.0,
    };
    const S_VEC: Vec3 = Vec3 {
        x: (-SQRT_3 / 2.0) as f32,
        y: 0.0,
        z: -0.5,
    };
    /// Get CloseNeighborhoodIterator, which iterates oder direct neighbors.
    pub fn get_neighbors(&self) -> CloseNeighborhoodIterator {
        CloseNeighborhoodIterator {
            coordinate: *self,
            index: 0,
        }
    }
    /// Get RingIterator with minimum Radius of 1.
    pub fn get_ring(&self, radius: usize) -> RingIterator {
        RingIterator::new(*self, radius)
    }
    /// Get NeighborhoodIterator with minimum Radius of 1.
    pub fn get_area(&self, radius: usize) -> NeighborhoodIterator {
        NeighborhoodIterator::new(*self, radius)
    }

    /// Get distance from this CubeCoordinate to another.
    pub fn distance(&self, coord: CubeCoordinate) -> usize {
        let diff = *self - coord;
        ((diff.q.abs() + diff.r.abs() + diff.s.abs()) / 2) as usize
    }
}

impl From<CubeCoordinate> for Vec3 {
    fn from(value: CubeCoordinate) -> Self {
        let (q, r, s) = (value.q as f32, value.r as f32, value.s as f32);
        (CubeCoordinate::Q_VEC * q + CubeCoordinate::R_VEC * r + CubeCoordinate::S_VEC * s)
            * EDGE_LENGTH as f32
    }
}

impl From<OffsetCoordinate> for CubeCoordinate {
    fn from(value: OffsetCoordinate) -> Self {
        let OffsetCoordinate { col, row } = value;
        let (q, r) = (col - (row - (row & 1)) / 2, row);
        CubeCoordinate { q, r, s: -q - r }
    }
}

impl From<CubeCoordinate> for OffsetCoordinate {
    fn from(value: CubeCoordinate) -> Self {
        let CubeCoordinate { q, r, .. } = value;
        OffsetCoordinate {
            col: q + (r - (r & 1)) / 2,
            row: r,
        }
    }
}
impl From<Vec3> for CubeCoordinate {
    fn from(value: Vec3) -> Self {
        let Vec3 { x, z, .. } = value;
        let q = (SQRT_3 / 3.0 * x as f64 - 1.0 / 3.0 * z as f64) / EDGE_LENGTH;
        let r = (2.0 / 3.0 * z as f64) / EDGE_LENGTH;
        let s = -q - r;
        CubeCoordinate::new_rounded(q as f32, r as f32, s as f32)
    }
}
impl From<Vec2> for CubeCoordinate {
    fn from(value: Vec2) -> Self {
        let Vec2 { x, y, .. } = value;
        let q = (SQRT_3 / 3.0 * x as f64 - 1.0 / 3.0 * y as f64) / EDGE_LENGTH;
        let r = (2.0 / 3.0 * y as f64) / EDGE_LENGTH;
        let s = -q - r;
        CubeCoordinate::new_rounded(q as f32, r as f32, s as f32)
    }
}

impl Add for CubeCoordinate {
    type Output = Self;

    fn add(self, other: Self) -> Self {
        Self {
            q: self.q + other.q,
            r: self.r + other.r,
            s: self.s + other.s,
        }
    }
}

impl AddAssign for CubeCoordinate {
    fn add_assign(&mut self, other: Self) {
        *self = *self + other;
    }
}

impl Sub for CubeCoordinate {
    type Output = Self;

    fn sub(self, other: Self) -> Self {
        Self {
            q: self.q - other.q,
            r: self.r - other.r,
            s: self.s - other.s,
        }
    }
}

pub fn lerp<R>(begin: R, end: R, t: f32) -> R
where
    R: Copy + Add<R, Output = R> + Mul<f32, Output = R>,
{
    begin * (1.0 - t) + end * t
}

/// Calculate shortest distance between ray and point.
pub fn distance_ray_point(ray: Ray3d, point: Vec3) -> f32 {
    (point - ray.origin).cross(*ray.direction).length() / ray.direction.length()
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::collections::HashSet;
    const RADIANS_30_SIN: f32 = 0.5; //(30.0 * PI / 180.0).sin();
    const RADIANS_30_COS: f32 = 0.866_025_403_784_438_6_f32; //(30.0 * PI / 180.0).cos();

    #[test]
    fn coordinate_conversion() {
        // test conversion offset->cube->offset
        for i in -10..10 {
            for j in -10..10 {
                let a = OffsetCoordinate { col: i, row: j };
                let b = CubeCoordinate::from(a);
                let c = OffsetCoordinate::from(b);
                assert_eq!(a, c);
            }
        }
        // test conversion cube->offset->cube
        for q in -10..10 {
            for r in -10..10 {
                for s in -10..10 {
                    //as long as convertion to canon representation not implemented
                    if q + r + s == 0 {
                        let a = CubeCoordinate { q, r, s };
                        let b = OffsetCoordinate::from(a);
                        let c = CubeCoordinate::from(b);
                        assert_eq!(a, c);
                    }
                }
            }
        }
    }

    #[test]
    /// checks whether to_coord conversion of offset and cube coordinates give same result within error margin
    fn to_coord() {
        let epsilon = 0.001;
        for i in -10..10 {
            for j in -10..10 {
                let a = OffsetCoordinate { col: i, row: j };
                let b: Vec3 = CubeCoordinate::from(a).into();
                let a: Vec3 = a.into();
                let c = (a - b).abs();
                assert!(c.x <= epsilon && c.z <= epsilon);
            }
        }
    }

    /// checks if various distances starting from a cube coordinate are still resulting in the same cube coordinate
    /// todo: check if distance out of bounds results in different cube coordinate
    #[test]
    fn from_vec_to_cube() {
        for q in -5..5 {
            for r in -5..5 {
                for s in -5..5 {
                    //as long as convertion to canon representation not implemented
                    if q + r + s == 0 {
                        let a = CubeCoordinate { q, r, s };
                        let b = Vec3::from(a);
                        // iterate vec3 with 30°
                        // todo: use edge_length from coordinates.rs or move global consts somewhere else?
                        const EDGE_LENGTH: f64 = 0.577_350_269_189_625_7_f64; // 1.0 / SQRT_3
                        const EPSILON: f64 = 0.05;
                        let mut in_circ = Vec3::new(0.0, 0.0, (0.5 - EPSILON) as f32);
                        let mut out_circ = Vec3::new(0.0, 0.0, (EDGE_LENGTH - EPSILON) as f32);
                        let mut small_circ = Vec3::new(0.0, 0.0, 0.2);

                        //test midpoint of tile
                        assert_eq!(CubeCoordinate::from(b), a);

                        // perform 12 iterations to go full circle
                        // test edges (flat top) of tiles
                        for _i in 1..12 {
                            let t = b + in_circ;
                            assert_eq!(CubeCoordinate::from(t), a);
                            let t = b + small_circ;
                            assert_eq!(CubeCoordinate::from(t), a);

                            rotate_vec_by_30_degrees(&mut in_circ);
                            rotate_vec_by_30_degrees(&mut small_circ);
                        }

                        // test corners (pointy tops) of tiles
                        for _i in 1..6 {
                            let t = b + out_circ;
                            assert_eq!(CubeCoordinate::from(t), a);
                            rotate_vec_by_30_degrees(&mut out_circ);
                            rotate_vec_by_30_degrees(&mut out_circ);
                        }
                    }
                }
            }
        }

        // used calculation from https://stackoverflow.com/questions/14607640/rotating-a-vector-in-3d-space
        // to rotate vec3 around y-axis
        fn rotate_vec_by_30_degrees(v: &mut Vec3) {
            let x = v.x;
            v.x = v.x * RADIANS_30_COS + v.z * RADIANS_30_SIN;
            v.z = -x * RADIANS_30_SIN + v.z * RADIANS_30_COS;
        }
    }

    #[test]
    fn coord_neighbor_iterators() {
        const TEST_SIZE: usize = 42;
        let mid = CubeCoordinate::new(0, 0, 0);
        let mut area_size = 0;
        for r in 1..=TEST_SIZE {
            let ring_iter = mid.get_ring(r);
            assert_eq!(ring_iter.size_hint().0, 6 * r);
            area_size += ring_iter.size_hint().0;
            let mut count_ring = 0;
            for coord in ring_iter {
                count_ring += 1;
                assert_eq!(
                    coord.q.abs() + coord.r.abs() + coord.s.abs(),
                    (r * 2) as isize
                );
            }
            assert_eq!(count_ring, 6 * r);
        }
        let area_iter = mid.get_area(TEST_SIZE);
        assert_eq!(area_iter.size_hint().0, area_size);

        let mut area_size_counted = 0;
        let mut used: HashSet<CubeCoordinate> = HashSet::new();
        let mut last = CubeCoordinate::new(0, 0, 0);
        for coord in area_iter {
            assert!(!used.contains(&coord));
            assert!(
                coord.q.abs() + coord.r.abs() + coord.s.abs()
                    >= last.q.abs() + last.r.abs() + last.s.abs()
            );
            last = coord;
            used.insert(coord);
            area_size_counted += 1;
        }
        assert_eq!(area_size, area_size_counted);
    }
}
