# 🛡 Campaign Mode

The campaign gamemode consists of multiple levels, the player can play.
Possible level ideas include tutorials or challenging levels with a starting
map.

Every level belongs to a group and every group consists of multiple levels,
with similar functionality or goals.
For example new tutorial levels should be placed in the `tutorial` group.

## 🎚 Creating a new level
> Every group consists of one or more levels, 
> which are playable by the user or a bot

Levels focus on the idea of a story line,
meaning that they define steps with goals,
which have to be reached in order to complete the level.

To make level definition easier, you can use the `define_level` macro:
```rust
define_level!(
<group-ref>::<level-ref>,
story: [
    // every level has a start segment,
    // the story will advance to the next segment once the goal has been met
    {
        goal: <run conditions separated using '=>'>,
        id: "<segment-id>",
        ref: <segment-ref>,
    },
   // every level can also have 0 or more middle segments, which are stepped through in sequentially
   // the next segment will be entered, once the goal of the current segment is met
     {
        goal: <run conditions separated using '=>'>,
        id: "<segment id>",
        ref: <segment-ref>,
    },
   // every level has an end segment
   // the level is considered finished, once it has been reached
    {
        id: "<segment-id>",
        ref: <segment-ref>,
    }
]);
```

After defining your levels story line, you can add Plugins to it,
by registering plugins 
[the same way you would do for a normal gamemode](./gamemodes.md).
The only difference being that the `game_mode`/parent state is the level group.
For example, the `game_mode` used for a camera tutorial level is `CampaignTutorial::CameraTutorial`

If you want to schedule systems for specific level segments,
you can use `<level-ref>::<segment-ref>`.

The macro itself should be placed in `levels/<group>/<level>.rs`.
Plugin and system definitions should go into the same file as `<level>Plugin`.
**Remember: you have to register your created plugins yourself,
but the level segment code is automatically registered by the groups macro**

## 🛍️ Creating a group
> A group is a collection of multiple levels

Similarly to levels, most of the group logic is generated using a macro.
```rust
campaign_groups!({
   "<group-id>": {
       ref: <group-ref>,
       levels: {
           "<level-id>": <level-ref>
       }
   } 
//...
});
```
As this macro can only be called once, you have to fill out the fields in the
`mod.rs` file in the `campaign` module.

The macro itself generates all the required code to initialize the levels.
Additionally, it also generates wrapper structs, the state tree and a
`init_campaign_mode` function,  which is used by the game to start a campagin
level.

## 🔮 Adding functionality
Adding functionality to a campaign level or group works the same way creating
new game modes does, but instead of using the `GameMode` as a parent state,
you use the campaign states. \
The [Game Mode Guide](./gamemodes.md) explains the creation process a little 
deeper.

## 🗨️ Translating
To be able to add text to the level segments, level- and group names,
you have to use the corresponding IDs:
- `level-<group-id>` = the translation ID of the group (i.e. Tutorial)
- `level-<group-id>-<level-id>` = the translation ID of the level
  (i.e. Camera)
- `level-<group-id>-<level-id>` = the translation ID of the segment 
  (the long-form explanation text)
  
## 🌳 State tree / graph

```mermaid
flowchart BT;

GM[GameMode]
GM_CP[Campaign]
GM_C[Challenge]
GM_CR[Creative]
GM_CP---GM
GM_C --- GM
GM_CR --- GM

CPG[CampaignGroups]
CPG_C[ChallengeExample]
CPG_T[Tutorial]
CPG_C --- CPG
CPG_T --- CPG

CPG --substate--> GM_CP

CLC[ChallengeCampaignGroup]
CLC_E[EasyLevel]
CLC_E --- CLC
CLC --substate--> CPG_C

CLP[CampaignLevelProgress]
CLP_I[Intro]
CLP_R[Running]
CLP_O[Outro]
CLP_I --- CLP
CLP_R --- CLP
CLP_O --- CLP
CLP --substate--> CPG

CCSW[CampaignChallengeStage]
CCSW_M[Meta]
CCSW_I[InGame]
CCSW_M --- CCSW
CCSW_I --- CCSW
CCSW --substate--> CLP_R
CCSW --substate--> CPG_C

CCS[CampaignChallengeState]
CCS_IR[InRound]
CCS_BR[BetweenRounds]
CCS_IR --- CCS
CCS_BR --- CCS

CCS --substate--> CCSW_I
```
