#!/usr/bin/perl
# Migraine Log - a simple multi-platform headache diary
# Copyright (C) 2025   Eskild Hustvedt
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
use 5.030;
use feature 'signatures';
no warnings 'experimental::signatures';
use Mojo::DOM;
use Mojo::JSON qw(decode_json encode_json);

# Reads a file into a scalar and returns it
sub slurp ($file) {
    open( my $in, '<', $file ) or die("Failed to open $file for reading: $!");
    local $/ = undef;
    my $content = <$in>;
    close($in);
    return $content;
}

# Converts a word into an equivalent-length randomized string, retaining
# punctuation, and replacing numbers with other, random, numbers, and keeping
# replacements consistent within a single run (so a single word will return the
# same random string for an entire file). You should probably be calling
# convertToGibberish instead of this.
sub makeGibberish($fromWord) {
    die("makeGibberish called with whitespace") if $fromWord =~ /\s/;
    state %builtWords;
    my $length  = length($fromWord);
    my $newWord = $builtWords{$fromWord} // '';
    my @word    = split( '', $fromWord );
    my @chars   = ( 'a' .. 'z' );
    my @numbers = ( 1 .. 9 );
    while ( length($newWord) < $length ) {
        my $currentCharacter = length($newWord);

        # We don't have 0..9 to avoid numbers *starting* with 0, which looks
        # odd. So from character 2 and onwards we add 0 back.
        if ( $currentCharacter == 1 ) {
            unshift( @numbers, 0 );
        }
        my $wordCharacter = $word[$currentCharacter];
        if ( $wordCharacter =~ /[\.\,\;\!\?]/ ) {
            $newWord .= $wordCharacter;
        }
        elsif ( $wordCharacter =~ /\d/ ) {
            $newWord .= $numbers[ rand scalar @numbers ];
        }
        else {
            $newWord .= $chars[ rand scalar @chars ];
        }
    }
    $builtWords{$fromWord} = $newWord;
    if ( $fromWord =~ /^[A-Z]/ ) {
        $newWord = ucfirst($newWord);
    }
    return $newWord;
}

# Converts a string of text into gibberish. This calls makeGibberish on any
# "word", where a word is any text separated by whitespace. It retains any
# whitespace as-is.
sub convertToGibberish($string) {
    my @entries;
    my @word;
    my $buildWord = sub {
        if (@word) {
            push( @entries, makeGibberish( join( '', @word ) ) );
            @word = ();
        }
    };
    foreach my $component ( split( '', $string ) ) {
        if ( $component =~ /\s/ ) {
            $buildWord->();
            push( @entries, $component );
        }
        else {
            push( @word, $component );
        }
    }
    $buildWord->();
    return scalar join( ' ', @entries );
}

sub main () {
    my $file = shift(@ARGV);
    if ( !defined $file || !-e $file || !-w $file ) {
        die(
"Usage: $0 file.html\n\nWill perform a rewrite of file.html, replacing certain personal details,\ncurrently notes and medication names, with placeholder gibberishn\nwhile retaining the data as a whole, as well as formatting and length.\nThe result is writteh to file.html.depersonalized"
        );
    }

    die("$file.depersonalized already exists. Move or delete it first.\n")
      if -e $file . ".depersonalized";
    my $patchFile = slurp($file);
    my $dom       = Mojo::DOM->new($patchFile);
    my $data      = $dom->find('#migraineLogData')->first->content;
    my $json      = decode_json($data);

    foreach my $month ( keys %{ $json->{data} } ) {
        foreach my $date ( keys %{ $json->{data}->{$month} } ) {
            if ( defined $json->{data}->{$month}->{$date}->{note} ) {
                $json->{data}->{$month}->{$date}->{note} = convertToGibberish(
                    $json->{data}->{$month}->{$date}->{note} );
            }
            if ( defined $json->{data}->{$month}->{$date}->{medicationList} ) {
                foreach my $med (
                    @{ $json->{data}->{$month}->{$date}->{medicationList} } )
                {
                    $med->{name} = convertToGibberish( $med->{name} );
                }
            }
        }
    }
    $dom->find('#migraineLogData')
      ->first->replace( '<script id="migraineLogData" type="text/json">'
          . encode_json($json)
          . '</script>' );
    open( my $outFile, '>', $file . '.depersonalized' )
      or die("Failed to open $file.depersonalized for writing: $!\n");
    print {$outFile} $dom->to_string;
}

main();
