// Migraine Log - a simple multi-platform headache diary
// Copyright (C) 2021-2025   Eskild Hustvedt
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

// This component will get embedded into exported HTML in order to parse the
// data and generate HTML.
//
// This is all rather suboptimal since the dart boilerplate that will get
// included will make this file, even minified and optimized, over 60k,
// significantly increasing the size of our exported file. What this DOES let
// us do, however, is test the html-rendering bits along with the rest of the
// app, which probably makes this a worthwhile tradeoff.

import 'dart:html';
import 'dart:convert';
import 'package:meta/meta.dart';
import 'dart:collection';
import 'definitions.dart';

String? debugHint;

class MigraineLogWebBase {
  final Map data;
  WebappStateManager? manager;

  MigraineLogWebBase({required this.data, this.manager});

  @visibleForTesting

  /// Converts a "month" string from data into a human-readable month string
  String getHumanMonthString(String month) {
    assert(data['strings']['months'][month] != null);
    return data['strings']['months'][month] ?? '?';
  }

  /// Converts a strength string (stringified int) into a strength message
  String? strength(String str) {
    var message = strengthBase(str);
    if (message != null && int.parse(str) > 0) {
      return message + ' (' + str + ')';
    }
    return message;
  }

  /// Converts a strength string (stringified int) into a strength message without a number
  String? strengthBase(String str) {
    var message = data['strings']['strength'][str];
    return message;
  }

  /// Retrieves a message by id, the message is provided by Migraine Log at
  /// export time for use here
  String message(String id) {
    assert(data['strings']['messages'][id] != null,
        'The message "$id" could not be found');
    return data['strings']['messages'][id] ?? id + '?';
  }

  /// Converts DateTime into a formatted, localized date string
  String formattedDateTime(DateTime dt) {
    Map? dateStrings = data['strings']['messages']['dateStrings'];
    assert(dateStrings != null);

    int wday = dt.weekday - 1; // dart starts the week on day 1, not 0
    int month = dt.month - 1; // dart starts months on 1, not 0
    String wdayLocalized = dateStrings!['weekdays'][wday];
    String monthLocalized = dateStrings['months'][month];
    return wdayLocalized +
        ', ' +
        dt.day.toString() +
        '. ' +
        monthLocalized +
        ' ' +
        dt.year.toString();
  }

  @visibleForTesting
  Map getMonthWithEmpty(String month) {
    var base = data['data'][month];
    Map withEmpty = {};
    var yearI = int.parse(month.substring(0, 4));
    var monthI = int.parse(month.substring(4, 6));
    var processDate = DateTime(yearI, monthI);
    while (processDate.month == monthI) {
      var dateStr = processDate.year.toString() +
          '-' +
          processDate.month.toString().padLeft(2, '0') +
          '-' +
          processDate.day.toString().padLeft(2, '0');
      if (base[dateStr] != null) {
        withEmpty[dateStr] = base[dateStr];
      } else {
        withEmpty[dateStr] = {
          "date": formattedDateTime(DateTime.parse(dateStr)),
          "strength": -100,
        };
      }
      processDate = processDate.add(Duration(days: 1));
      // Don't list dates after today
      if (processDate.isAfter(DateTime.now())) {
        break;
      }
    }
    return withEmpty;
  }

  @visibleForTesting

  /// Retrieves a single month by its string representation
  Map? getMonth(String month) {
    assert(data['data'][month] != null);
    if (manager != null && manager!.displayEmptyDays) {
      return getMonthWithEmpty(month);
    }
    return data['data'][month];
  }

  @visibleForTesting

  /// Retrieves a single month's summary by its string representation
  Map? getMonthSummary(String month) {
    return data['summaries'][month];
  }

  /// Builds a stringified list of medications
  String medicationListToStr(List? list) {
    if (list != null && list.isNotEmpty) {
      String msg = '';
      for (var entry in list) {
        if (entry['timeString'] != '-') {
          msg += entry['timeString'] + ' ';
        }
        msg += entry['name'] + "\n";
      }
      return msg;
    }
    return message('none');
  }

  /// Sorts the list toSort by the value of `field` in the `source` map
  List sortByField(List toSort, Map source, String field) {
    if (manager!.direction == sortDirection.desc) {
      toSort.sort((a, b) =>
          getField(source[b], field).compareTo(getField(source[a], field)));
    } else {
      toSort.sort((a, b) =>
          getField(source[a], field).compareTo(getField(source[b], field)));
    }
    return toSort;
  }

  /// Returns the `field` in `source`. If it is null it returns an empty
  /// string. If it is a list then it gets converted to a String. If it is a
  /// String or int then it gets returned directly.
  dynamic getField(Map source, String field) {
    var value = source[field];
    if (value == null) {
      return '';
    }
    assert(value is String || value is List || value is int);
    if (value is String) {
      return value;
    } else if (value is int) {
      return value;
    } else if (value is List) {
      return value.join('');
    }
  }

  /// Sorts a list in descending order
  List sorted(List thing) {
    thing.sort((a, b) => b.compareTo(a));
    return thing;
  }

  Element tableElement({small = false}) {
    Element table = Element.table();
    if (small != true) {
      table.classes.add("table");
    }
    if (manager!.colorizeTable) {
      table.classes.add('colorized');
    }
    return table;
  }
}

class WebappStateManager {
  void listen(Function cb) {
    cbs.add(cb);
  }

  void setTableSort(sortField setField, sortDirection setDirection) {
    field = setField;
    direction = setDirection;
    notify();
  }

  void notify() {
    for (var cb in cbs) {
      cb();
    }
  }

  @visibleForTesting
  List<Function> cbs = [];

  /// The current sort field
  @visibleForTesting
  sortField field = sortField.date;

  /// The current sort direction
  @visibleForTesting
  sortDirection direction = sortDirection.desc;

  /// The current display mode
  displayMode mode = displayMode.monthlyList;

  /// If we're to display empty days in the month or not
  bool displayEmptyDays = false;

  /// If the text in the table should be colorized by pain strength or not
  bool colorizeTable = false;

  bool showSummaryColumns = true;

  /// The limit (in number of months) to apply to the yearly list table
  /// A negative number is treated as "everything"
  int yearlyListLimit = 12;
}

/// The display mode
enum displayMode { monthlyList, twelveMonthSummary }

/// The field to sort a table by
enum sortField { date, strength, medication, note }

/// The sorting direction
enum sortDirection { desc, asc }

/*
   Builds a single month table
 */
class MonthTable extends MigraineLogWebBase {
  MonthTable({
    required this.strMonth,
    required super.manager,
    required super.data,
  }) {
    manager!.listen(() => build());
  }

  /// The string representation of the month we're rendering
  final String strMonth;

  /// Our current container (table-element). This changes throughout the
  /// lifetime of the object.
  @visibleForTesting
  Element? container;

  /// A map of sortFields to their string key equivalents. Used to be able to
  /// map a sort direction to a field contained in the data.
  Map fieldToKey = {
    sortField.note: 'note',
    sortField.medication: 'medicationList',
    sortField.strength: 'strength',
  };

  /// Returns the current sorting arrow for the selected field, as an Element.
  /// This element will always contain the arrow for the current direction, but
  /// unless forField is the current `field` then it will have its visibility
  /// set to hidden. This is so that it takes up space in the dom so text
  /// position doesn't jump around by having the arrow added and removed.
  Element sortingArrow(sortField forField) {
    String sorter = ' ' + (manager!.direction == sortDirection.asc ? '▲' : '▼');
    var entry = Element.span()
      ..innerText = sorter
      ..className = "sortArrow";
    if (forField != manager!.field) {
      entry.style.visibility = 'hidden';
    }
    return entry;
  }

  /// Builds a single table header (th-element) and hooks up onClick listeners
  /// for changing the sort direction, looks up the message string for the
  /// header and inserts a sorting arrow
  Element tableHeader(String msgField, sortField sortBy) {
    return Element.th()
      ..innerText = message(msgField)
      ..append(sortingArrow(sortBy))
      ..className = 'like-link'
      ..onClick.listen((_) => changeSort(sortBy));
  }

  /// Retrieves a list of keys in the month strMonth that is sorted by the
  /// currently selected field and direction
  List sortedMonthKeys() {
    Map month = getMonth(strMonth)!;
    List sorted = month.keys.toList();

    /// Date is special, since we use the key itself, so sort that here.
    if (manager!.field == sortField.date) {
      if (manager!.direction == sortDirection.desc) {
        sorted.sort((a, b) => b.compareTo(a));
      } else {
        sorted.sort((a, b) => a.compareTo(b));
      }
    } else if (manager!.field == sortField.strength) {
      sorted = sortByField(sorted, month, fieldToKey[manager!.field]);
    } else {
      assert(fieldToKey[manager!.field] != null);
      sorted = sortByField(sorted, month, fieldToKey[manager!.field]);
    }
    return sorted;
  }

  /// Changes the sorting field and sets the direction to desc, or, if the
  /// current field is the same field we're changing to, changes the sorting
  /// direction. In either case this will trigger a build().
  void changeSort(sortField changeToField) {
    if (changeToField == manager!.field) {
      manager!.setTableSort(
          changeToField,
          manager!.direction == sortDirection.asc
              ? sortDirection.desc
              : sortDirection.asc);
    } else {
      manager!.setTableSort(changeToField, sortDirection.desc);
    }
  }

  /// Builds this table and injects it into the HTML. This is also used to
  /// rebuild the table when something changes, like the sorting, in which case
  /// it will replace the table already in the DOM with the updated table.
  Element build() {
    Element table = tableElement();
    Map? month = getMonth(strMonth);
    table.append(
      Element.tag('thead')
        ..append(
          Element.tr()
            ..append(tableHeader('date', sortField.date))
            ..append(tableHeader('strength', sortField.strength))
            ..append(tableHeader('takenMeds', sortField.medication))
            ..append(tableHeader('note', sortField.note)),
        ),
    );
    var tbody = Element.tag('tbody');
    table.append(tbody);
    for (var date in sortedMonthKeys()) {
      Map day = month![date];
      String note = "";
      if (day['note'] != null) {
        note = day['note'];
      }
      tbody.append(
        Element.tr()
          ..setAttribute('strength', day['strength'])
          ..append(Element.td()..innerText = day['date'])
          ..append(Element.td()
            ..innerText = day['strength'] != -100
                ? strength(day['strength'].toString())!
                : '')
          ..append(Element.td()
            ..innerText = medicationListToStr(day['medicationList']))
          ..append(Element.td()..innerText = note),
      );
    }
    if (container != null) {
      container!.replaceWith(table);
    }
    container = table;
    return table;
  }

  /// Retrieves the root element for this table. Typically a Element.table
  Element? get element {
    if (container == null) {
      build();
    }
    return container;
  }
}

/*
   Builds the entire "summary table" view (ie. the 12 month table).
 */
class SummaryTableBuilder extends MigraineLogWebBase {
  Map<String, String> medicationMap = {};

  SummaryTableBuilder({
    required super.manager,
    required super.data,
  }) {
    manager!.listen(() => build());
    constructMedicationMap();
  }

  Element monthCell(Map content) {
    String text = "";
    Element medications = Element.tag("sup")..className = 'text-xs';
    int str = 0;
    if (content["strength"] != null && content["strength"] > 0) {
      text = content["strength"].toString();
      str = content["strength"];
    }
    var medList = content["medicationList"];
    if (medList != null && medList.isNotEmpty) {
      final meds = SplayTreeSet<String>((a, b) => a.compareTo(b));
      for (var entry in medList) {
        if (entry["name"] is String) {
          meds.add(entry["name"]);
        }
      }
      String medStr = '';
      for (var med in meds) {
        medStr += medicationMap[med]!;
      }
      medications.innerText = medStr;
    }
    return Element.td()
      ..innerText = text
      ..setAttribute('strength', str)
      ..append(medications);
  }

  /// Injects summary columns into parent, summarizing the number of days for
  /// each medication and strength
  void monthTableSummary(String strMonth, Element parent) {
    Map<String, int> medicationDays = {};
    Map<int, int> strengthDays = {};
    Map month = getMonth(strMonth)!;
    for (var entry in month.keys.toList()) {
      var day = month[entry];
      if (day['strength'] != null) {
        int strength = day['strength'];
        strengthDays[strength] = (strengthDays[strength] ?? 0) + 1;
      }
      var medList = day["medicationList"];
      if (medList != null && medList.isNotEmpty) {
        Set medsSeen = {};
        for (var entry in medList) {
          var name = entry["name"];
          if (name is String && !medsSeen.contains(name)) {
            medsSeen.add(name);
            medicationDays[name] = (medicationDays[name] ?? 0) + 1;
          }
        }
      }
    }

    parent.append(Element.td()..className = 'summary-padding-cell');

    for (var med in getMedicationsInUseInPeriod()) {
      int days = medicationDays[med] ?? 0;
      parent.append(Element.td()..innerText = days.toString());
    }
    for (var strength = 1; strength <= 3; strength++) {
      int days = strengthDays[strength] ?? 0;
      parent.append(Element.td()..innerText = days.toString());
    }
  }

  /// Builds a single table row for the summary table (representing one month)
  Element monthTableRow(String strMonth) {
    Map month = getMonthWithEmpty(strMonth);
    List sorted = month.keys.toList();
    Element row = Element.tr();
    row.append(Element.td()..innerText = getHumanMonthString(strMonth));
    // Note: we actually want 31 columns, but the array is 0-indexed
    for (int no = 0; no <= 30; no++) {
      if (no < sorted.length) {
        var entry = sorted[no];
        row.append(monthCell(month[entry]));
      } else {
        row.append(Element.td()..className = 'bg-faded');
      }
    }
    if (manager!.showSummaryColumns) {
      monthTableSummary(strMonth, row);
    }
    return row;
  }

  /// Retrieves a list of all medications that have been *in use* in the
  /// selected time period. This can be different from the total number of
  /// medications that exist.
  List<String> getMedicationsInUseInPeriod() {
    Map months = data['data'];
    Set<String> meds = {};
    int monthNo = 0;
    for (var month in sorted(months.keys.toList())) {
      monthNo++;
      for (var day in months[month].keys.toList()) {
        var medList = months[month][day]["medicationList"];
        if (medList != null && medList.isNotEmpty) {
          for (var entry in medList) {
            if (entry["name"] is String) {
              meds.add(entry["name"]);
            }
          }
        }
      }
      if (monthNo >= manager!.yearlyListLimit && manager!.yearlyListLimit > 0) {
        break;
      }
    }
    return meds.toList()
      ..sort((a, b) => medicationMap[a]!.compareTo(medicationMap[b]!));
  }

  /// Builds the table header
  Element buildHeader() {
    Element th = Element.tr();
    // Empty node for the month name
    th.append(Element.th());
    for (var d = 1; d <= 31; d++) {
      th.append(Element.th()..innerText = d.toString());
    }
    if (manager!.showSummaryColumns) {
      th.append(Element.th()..className = 'summary-padding-cell');
      for (var med in getMedicationsInUseInPeriod()) {
        th.append(Element.th()..innerText = medicationMap[med]! + '¹');
      }
      for (var strength = 1; strength <= 3; strength++) {
        th.append(Element.th()..innerText = strength.toString() + '²');
      }
    }
    return th;
  }

  /// Creates a map of medications -> shorthand. It tries to ensure that the
  /// assigned characters stay consistent over time, so that the earliest
  /// medication registered will always be "A"
  void constructMedicationMap() {
    Map months = data['data'];
    Set<String> meds = {};
    List<String> possibleKeys = [
      'A',
      'B',
      'C',
      'D',
      'E',
      'F',
      'G',
      'H',
      'I',
      'J',
      'K',
      'L',
      'M',
      'N',
      'O',
      'P',
      'Q',
      'R',
      'S',
      'T',
      'U',
      'V',
      'W',
      'X',
      'Y',
      'Z'
    ];
    for (var month in months.keys.toList()) {
      for (var day in months[month].keys.toList()) {
        var medList = months[month][day]["medicationList"];
        if (medList != null && medList.isNotEmpty) {
          for (var entry in medList) {
            if (entry["name"] is String) {
              meds.add(entry["name"]);
            }
          }
        }
      }
    }
    List<String> sortedMeds = meds.toList();
    sortedMeds.sort();
    bool useAlpha = true;
    if (sortedMeds.length > possibleKeys.length) {
      useAlpha = false;
    }

    for (int i = 0; i < sortedMeds.length; i++) {
      String key;
      if (useAlpha) {
        key = possibleKeys[i];
      } else {
        key = i.toString();
      }
      medicationMap[sortedMeds[i]] = key;
    }
  }

  /// Builds a table of all medications
  Element medsTable() {
    Element table = Element.table();
    final sorted = getMedicationsInUseInPeriod();
    for (var medication in sorted) {
      table.append(Element.tr()
        ..append(Element.td()..innerText = medicationMap[medication]!)
        ..append(Element.td()..innerText = medication));
    }
    return table;
  }

  /// Builds a table of all strengths
  Element strengthTable() {
    Element table = tableElement(small: true);
    final strengths = [strengthBase("1"), strengthBase("2"), strengthBase("3")];
    for (var i = 0; i < strengths.length; i++) {
      table.append(Element.tr()
        ..append(Element.td()
          ..innerText = (i + 1).toString()
          ..attributes['strength'] = (i + 1).toString())
        ..append(Element.td()
          ..innerText = strengths[i]!
          ..attributes['strength'] = (i + 1).toString()));
    }
    return table;
  }

  /// Builds the summary component
  Element build() {
    Element wrapper = Element.div()
      ..className = 'text-sm'
      // Force landscape printing for this mode, and set the print margins to 0
      ..append(Element.tag('style')
        ..attributes['type'] = 'text/css'
        ..innerText = '@media print{@page {size: landscape; margin:0;}}');

    Element tableWrapper = Element.div()..className = 'mb-1';
    Element table = tableElement()
      ..classes.addAll(['text-sm', 'reduced-padding', 'mb-0']);
    table.append(buildHeader());
    tableWrapper.append(table);

    if (manager!.showSummaryColumns) {
      tableWrapper.append(Element.div()
        ..className = 'text-xs'
        ..innerText = '¹) ' + message('numberOfDaysMedicationDescription'));
      tableWrapper.append(Element.div()
        ..className = 'text-xs'
        ..innerText = '²) ' + message('numberOfDaysStrengthDescription'));
    }
    Map months = data['data'];
    int monthNo = 0;
    for (var entry in sorted(months.keys.toList())) {
      monthNo++;
      table.append(monthTableRow(entry));
      if (monthNo >= manager!.yearlyListLimit && manager!.yearlyListLimit > 0) {
        break;
      }
    }
    return wrapper
      ..append(tableWrapper)
      ..append(Element.div()
        ..className = 'overflow-auto'
        ..append(Element.div()
          ..className = 'float-left'
          ..append(medsTable()))
        ..append(Element.div()
          ..className = 'float-left ml-1'
          ..append(strengthTable())));
  }
}

class dataToHTML extends MigraineLogWebBase {
  Element? container;

  List<String> hiddenMonths = [];
  dataToHTML({
    required super.data,
  }) {
    manager ??= WebappStateManager();
  }

  int monthLimit = -1;

  /// Generates DOM elements that represent the summary of a single month
  Element monthSummary(String strMonth) {
    Map month = getMonthSummary(strMonth)!;
    Element table = Element.table();
    for (var type in month['specificSummaries'].keys) {
      table.append(Element.tr()
        ..append(Element.td()..innerText = type)
        ..append(Element.td()..innerText = month['specificSummaries'][type]));
    }
    if (month["medicationDays"] != "0") {
      table.append(Element.tr()
        ..append(Element.td()..innerText = message("takenMeds"))
        ..append(Element.td()..innerText = month['medicationDays']));
    }
    table.append(Element.tr()
      ..append(Element.td()..innerText = message("totalHeadacheDays"))
      ..append(Element.td()..innerText = month['headacheDays']));
    return table;
  }

  /// Retrieves the "header" for a month.
  /// This also handles hiding/showing individual months.
  Element monthHeader(String strMonth) {
    var container = Element.tag('h2')
      ..innerText = getHumanMonthString(strMonth);
    container.append(Element.tag('span')
      ..innerText = '[' + message('hide') + ']'
      ..className = 'hide-month'
      ..title = message('hideTooltip')
      ..onClick.listen((_) {
        var month = container.closest('.month')!;
        var replacement = Element.div()
          ..innerText = '(' +
              message('hidden') +
              ' ' +
              getHumanMonthString(strMonth) +
              ')'
          ..className = 'hidden-month';
        replacement.onClick.listen((_) {
          replacement.replaceWith(month);
        });
        month.replaceWith(replacement);
        container.closest('.month')!.remove();
      }));
    return container;
  }

  /// Generates DOM elements that represent a single month
  Element monthToHTML(String strMonth) {
    Element container = Element.div()..className = 'month';
    container.append(monthHeader(strMonth));
    Element table =
        MonthTable(data: data, strMonth: strMonth, manager: manager!).element!;
    container
      ..append(monthSummary(strMonth))
      ..append(table);
    return container;
  }

  /// Firefox has a bug where table borders disappear in the print preview when
  /// they're 1px in width, so we double it for printing on firefox. This is a
  /// ugly hack, but needed to work around the firefox bug.
  void addFirefoxHack(Element to) {
    if (window.navigator.userAgent.contains('Firefox')) {
      to.append(Element.tag('style')
        ..innerText = '@media print { th, td {border: 2px solid black} }'
        ..attributes['type'] = 'text/css');
    }
  }

  void rebuild() {
    var oldContainer = container;
    try {
      container = Element.div();
      constructUI();
      if (oldContainer != null) {
        oldContainer.replaceWith(container!);
      }
    } catch (e) {
      if (oldContainer != null) {
        oldContainer.replaceWith(fatalErrorMessage(e));
      } else {
        // Mostly in tests
        rethrow;
      }
    }
  }

  void monthlyListFilterDropdown(Element parent) {
    Map months = data['data'];
    var filterSelect = Element.tag('select')
      ..onChange.listen((ev) {
        var selected = int.parse((ev.currentTarget as SelectElement).value!);
        if (selected != monthLimit) {
          monthLimit = selected;
          rebuild();
        }
      });
    filterSelect.append(Element.tag('option')
      ..attributes['id'] = 'monthNoFilter'
      ..attributes['value'] = "-100"
      ..innerText = message('everything'));
    var filterLimit = 3;
    while (filterLimit < months.length) {
      filterSelect.append(Element.tag('option') as OptionElement
        ..selected = monthLimit == filterLimit
        ..attributes['id'] = 'monthNoFilter'
        ..attributes['value'] = filterLimit.toString()
        ..innerText = filterLimit.toString() + ' ' + message('months'));

      if (filterLimit >= 6) {
        filterLimit += 3;
      } else {
        filterLimit += 1;
      }
    }
    if (months.length > 3) {
      parent
        ..append(Element.tag('label')
          ..attributes['for'] = 'monthNoFilter'
          ..innerText = message('show') + ' ')
        ..append(filterSelect);
    }
  }

  void yearlyListFilterDropdown(Element parent) {
    Map months = data['data'];
    var filterSelect = Element.tag('select')
      ..onChange.listen((ev) {
        var selected = int.parse((ev.currentTarget as SelectElement).value!);
        if (selected != manager!.yearlyListLimit) {
          manager!.yearlyListLimit = selected;
          rebuild();
        }
      });
    filterSelect.append(Element.tag('option')
      ..attributes['id'] = 'monthNoFilter'
      ..attributes['value'] = "-100"
      ..innerText = message('everything'));
    var filterLimit = 12;
    while (filterLimit < months.length) {
      filterSelect.append(Element.tag('option') as OptionElement
        ..selected = manager!.yearlyListLimit == filterLimit
        ..attributes['id'] = 'monthNoFilter'
        ..attributes['value'] = filterLimit.toString()
        ..innerText = filterLimit.toString() + ' ' + message('months'));

      filterLimit += 3;
    }
    if (months.length > 12) {
      parent
        ..append(Element.tag('label')
          ..attributes['for'] = 'monthNoFilter'
          ..innerText = message('show') + ' ')
        ..append(filterSelect);
    }
  }

  Element buildFilterUI() {
    var filterContainer = Element.div()
      ..className = 'hidden-on-print mb-1'
      ..append(Element.tag('b')..innerText = message('filter') + ' ');
    var filterCheckbox = Element.tag('input')
      ..attributes['type'] = 'checkbox'
      ..attributes['id'] = 'emptyDaysCheckbox'
      ..className = 'ml-1'
      ..onChange.listen((ev) {
        manager!.displayEmptyDays = !manager!.displayEmptyDays;
        rebuild();
      });
    var filterText = Element.tag('label')
      ..attributes['for'] = 'emptyDaysCheckbox'
      ..text = message('listEmptyDays');
    if (manager!.displayEmptyDays) {
      filterCheckbox.attributes['checked'] = 'checked';
    }

    if (manager!.mode == displayMode.monthlyList) {
      monthlyListFilterDropdown(filterContainer);
    } else if (manager!.mode == displayMode.twelveMonthSummary) {
      yearlyListFilterDropdown(filterContainer);
    }
    var colorizeCheckbox = Element.tag('input')
      ..attributes['type'] = 'checkbox'
      ..attributes['id'] = 'colorizeTableCheckbox'
      ..className = 'ml-1'
      ..onChange.listen((ev) {
        manager!.colorizeTable = !manager!.colorizeTable;
        rebuild();
      });
    var colorizeText = Element.tag('label')
      ..attributes['for'] = 'colorizeTableCheckbox'
      ..text = message('colorizeTable');
    if (manager!.colorizeTable) {
      colorizeCheckbox.attributes['checked'] = 'checked';
    }
    if (manager!.mode == displayMode.monthlyList) {
      filterContainer.append(filterCheckbox);
      filterContainer.append(filterText);
    }
    return filterContainer
      ..append(colorizeCheckbox)
      ..append(colorizeText);
  }

  void buildSelectionUI() {
    var mode = Element.div()
      ..innerText = message("displayMode") + ': '
      ..className = 'mb-1 hidden-on-print';

    var twelveMonthSummaryInput = Element.tag('input')
      ..attributes['type'] = 'radio'
      ..attributes['id'] = 'displayModeSummaryTable'
      ..attributes['name'] = 'displayModeSummaryTable'
      ..className = 'ml-1'
      ..onChange.listen((ev) {
        manager!.mode = displayMode.twelveMonthSummary;
        rebuild();
      });
    var twelveMonthSummaryLabel = Element.tag('label')
      ..attributes["for"] = "displayModeSummaryTable"
      ..innerText = message("displayModeSummaryTable");
    var monthlyListSummaryInput = Element.tag('input')
      ..attributes['type'] = 'radio'
      ..attributes['id'] = 'displayModeMonthlyList'
      ..attributes['name'] = 'displayModeMonthlyList'
      ..className = 'ml-1'
      ..onChange.listen((ev) {
        manager!.mode = displayMode.monthlyList;
        rebuild();
      });
    var monthlyListSummaryLabel = Element.tag('label')
      ..attributes["for"] = "displayModeMonthlyList"
      ..innerText = message("displayModeMonthlyList");
    if (manager!.mode == displayMode.monthlyList) {
      monthlyListSummaryInput.attributes["checked"] = "checked";
    } else if (manager!.mode == displayMode.twelveMonthSummary) {
      twelveMonthSummaryInput.attributes["checked"] = "checked";
    }
    mode
      ..append(monthlyListSummaryInput)
      ..append(monthlyListSummaryLabel)
      ..append(twelveMonthSummaryInput)
      ..append(twelveMonthSummaryLabel);
    container!.append(mode);
  }

  void constructUI() {
    buildSelectionUI();
    container!.append(buildFilterUI());
    if (manager!.mode == displayMode.twelveMonthSummary) {
      constructSummaryTableUI();
    } else {
      constructMonthlyUI();
    }
  }

  void constructSummaryTableUI() {
    SummaryTableBuilder builder =
        SummaryTableBuilder(manager: manager, data: data);
    container!.append(builder.build());
  }

  void constructMonthlyUI() {
    Map months = data['data'];
    var monthNo = 1;
    for (var entry in sorted(months.keys.toList())) {
      container!.append(monthToHTML(entry));
      if (monthLimit != -1 && ++monthNo > monthLimit) {
        break;
      }
    }
  }

  /// Builds our UI
  Element build() {
    addFirefoxHack(querySelector('head')!);
    rebuild();
    return container!;
  }
}

Element fatalErrorMessage(dynamic e) {
  Element message = Element.div()
    ..setInnerHtml(
        "Migraine Log encountered a fatal error while building the interface.<br />Your data is still stored in the file, but Migraine Log is not able to display it here (you can, however, import it into the app).&nbsp;")
    ..append(Element.a()
      ..setAttribute(
          'href', "https://gitlab.com/mglog/org.zerodogg.migraineLog/issues")
      ..innerText = 'Please report this bug.');
  String errorCode = '[' + e.runtimeType.toString() + '] ' + e.toString();
  Element error = Element.div()
    ..append(message)
    ..append(Element.span()..innerText = 'Error code: ')
    ..append(Element.tag('code')..innerText = errorCode);
  if (e is Error) {
    error.append(Element.tag('pre')..innerText = e.stackTrace.toString());
  }
  if (debugHint != null) {
    error.append(Element.div()..innerText = "Debug hint: $debugHint");
  }
  return error;
}

void main() {
  // The loading element
  var loadingElement = querySelector('#loading')!;

  try {
    // The JSON data is in an element with the ID migraineLogData
    var jsonElement = querySelector('#migraineLogData')!;

    // Decode the data
    var data = jsonDecode(jsonElement.innerText);

    if (data["exportVersion"] != DATAVERSION) {
      debugHint = "exportVersion " +
          data["exportVersion"].toString() +
          " and DATAVERSION " +
          DATAVERSION.toString() +
          " do not match";
    }

    // Perform the UI build
    var dh = dataToHTML(data: data);

    // Replace the loading element with the data we just built
    loadingElement.replaceWith(dh.build());
  } catch (e) {
    loadingElement.replaceWith(fatalErrorMessage(e));
  }
}
