/*
 * MainActivity.kt
 * Implements the MainActivity class
 * The Main Activity hosts the MainFragment
 *
 * This file is part of
 * STAY PUT - Unplug Alert App
 *
 * Copyright (c) 2022-25 - Y20K.org
 * Licensed under the MIT-License
 * http://opensource.org/licenses/MIT
 */


package org.y20k.stayput

import android.os.Bundle
import android.view.View
import android.widget.FrameLayout
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updateLayoutParams
import androidx.navigation.fragment.NavHostFragment
import androidx.navigation.ui.AppBarConfiguration
import androidx.navigation.ui.navigateUp
import com.google.android.material.color.DynamicColors


/*
 * MainActivity class
 */
class MainActivity: AppCompatActivity() {

    /* Define log tag */
    private val TAG: String = MainActivity::class.java.simpleName


    /* Main class variables */
    private lateinit var appBarConfiguration: AppBarConfiguration


    /* Overrides onCreate from AppCompatActivity */
    override fun onCreate(savedInstanceState: Bundle?) {
        enableEdgeToEdge()
        DynamicColors.applyToActivityIfAvailable(this) // must be the first line
        super.onCreate(savedInstanceState)
        // set up views
        setContentView(R.layout.activity_main)
        // set up edge to edge display
        val rootView: View = findViewById<View>(R.id.root_view)
        setupEdgeToEdge(rootView)
        // turn screen off and keyguard on
        if (intent.hasExtra(Keys.EXTRA_SHOW_ACTIVITY_ON_LOCKSCREEN) && intent.getBooleanExtra(Keys.EXTRA_SHOW_ACTIVITY_ON_LOCKSCREEN, false) == true) {
            setShowWhenLocked(true)
            setTurnScreenOn(true)
        }
    }


    /* Overrides onDestroy from AppCompatActivity */
    override fun onDestroy() {
        super.onDestroy()
        // turn screen off and keyguard on
        setShowWhenLocked(false)
        setTurnScreenOn(false)
    }


    /* Overrides onSupportNavigateUp from AppCompatActivity */
    override fun onSupportNavigateUp(): Boolean {
        // Taken from: https://developer.android.com/guide/navigation/navigation-ui#action_bar
        val navHostFragment = supportFragmentManager.findFragmentById(R.id.main_host_container) as NavHostFragment
        val navController = navHostFragment.navController
        return navController.navigateUp(appBarConfiguration) || super.onSupportNavigateUp()
    }


    /* Sets up margins/paddings for edge to edge view */
    private fun setupEdgeToEdge(view: View) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { v, insets ->
            // get measurements for status and navigation bar
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars() or WindowInsetsCompat.Type.displayCutout())
            // apply measurements
            v.updateLayoutParams<FrameLayout.LayoutParams> {
                topMargin = systemBars.top
                bottomMargin = systemBars.bottom
            }
            // return the insets
            insets
        }
    }

}
