/*
 * PodcastFragmentLayoutHolder.kt
 * Implements the PodcastFragmentLayoutHolder class
 * A PodcastFragmentLayoutHolder hold references to the views of the PodcastFragment
 *
 * This file is part of
 * ESCAPEPOD - Free and Open Podcast App
 *
 * Copyright (c) 2018-25 - Y20K.org
 * Licensed under the MIT-License
 * http://opensource.org/licenses/MIT
 */


package org.y20k.escapepod.ui

import android.annotation.SuppressLint
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.graphics.drawable.Drawable
import android.os.Build
import android.os.Vibrator
import android.view.View
import android.view.ViewTreeObserver.OnPreDrawListener
import android.widget.Filterable
import android.widget.ImageView
import android.widget.Toast
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.app.ActivityCompat.startPostponedEnterTransition
import androidx.core.graphics.Insets
import androidx.core.net.toUri
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.core.view.updateLayoutParams
import androidx.core.view.updatePadding
import androidx.fragment.app.FragmentActivity
import androidx.recyclerview.widget.DefaultItemAnimator
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.LinearSmoothScroller
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.SimpleItemAnimator
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.google.android.material.appbar.MaterialToolbar
import com.google.android.material.textview.MaterialTextView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers.IO
import kotlinx.coroutines.Dispatchers.Main
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.y20k.escapepod.BaseMainActivity
import org.y20k.escapepod.Keys
import org.y20k.escapepod.R
import org.y20k.escapepod.database.CollectionDatabase
import org.y20k.escapepod.database.objects.Podcast
import org.y20k.escapepod.database.objects.PodcastDescription
import org.y20k.escapepod.dialogs.DescriptionDialog
import org.y20k.escapepod.extensions.setCleanHtml
import org.y20k.escapepod.helpers.DownloadHelper
import org.y20k.escapepod.helpers.UiHelper


/*
 * LayoutHolder class
 */
data class PodcastFragmentLayoutHolder(val rootView: View, val collectionDatabase: CollectionDatabase, val filtered: Boolean) {

    /* Define log tag */
    private val TAG: String = PodcastFragmentLayoutHolder::class.java.simpleName


    /* Interface for list events */
    interface EpisodesListDragListener {
        fun onEpisodesListDragStateChanged(newState: Int)
    }


    /* Main class variables */
    private lateinit var systemBars: Insets
    val swipeRefreshLayout: SwipeRefreshLayout
    val toolbar: MaterialToolbar
    val episodesList: RecyclerView
    val filterIcon: ImageView
    val filterText: MaterialTextView
    val coverView: ImageView
    var episodesListDragListener: EpisodesListDragListener? = null
    lateinit var podcast: Podcast
    lateinit var podcastDescription: PodcastDescription
    private val layoutManager: LinearLayoutManager
    private val scroller: RecyclerView.SmoothScroller
    private val podcastDescriptionView: MaterialTextView
    private val podcastDescriptionFadeView: View
    private val podcastWebsiteView: MaterialTextView
    private val podcastLinksDivider: ImageView
    private val podcastFeedView: MaterialTextView
    private var podcastCoverUriString: String = String()


    /* Init block */
    init {
        // find views
        swipeRefreshLayout = rootView.findViewById(R.id.swipe_refresh_podcast_layout)
        toolbar = rootView.findViewById<MaterialToolbar>(R.id.toolbar)
        episodesList = rootView.findViewById(R.id.episodes_list)
        filterIcon = rootView.findViewById(R.id.filter_icon)
        filterText = rootView.findViewById(R.id.filter_text)
        coverView = rootView.findViewById(R.id.podcast_cover)
        podcastDescriptionView = rootView.findViewById(R.id.podcast_description)
        podcastDescriptionFadeView = rootView.findViewById(R.id.podcast_description_fade)
        podcastWebsiteView = rootView.findViewById(R.id.podcast_website)
        podcastLinksDivider = rootView.findViewById(R.id.divider_centered_dot)
        podcastFeedView = rootView.findViewById(R.id.podcast_feed)

        // set up filter
        if (filtered) {
            filterText.text = rootView.context.getText(R.string.episode_list_filter_downloaded)
        } else {
            filterText.text = rootView.context.getText(R.string.podcast_list_button_all_episodes)
        }

        // set up the description fade
        // TODO: https://stackoverflow.com/questions/46642245/ellipsize-text-in-textview-without-specifying-maxlines
        podcastDescriptionFadeView.background = UiHelper.createColorTransparencyGradientDrawable(rootView, com.google.android.material.R.attr.colorSurface, 0.1f)

        // set up RecyclerView
        layoutManager = CustomLayoutManager(rootView.context)
        episodesList.layoutManager = layoutManager
        val animator: SimpleItemAnimator = DefaultItemAnimator()
        animator.supportsChangeAnimations = true
        episodesList.itemAnimator = animator
        episodesList.setHasFixedSize(true)

        // customize scroller
        scroller = object : LinearSmoothScroller(rootView.context) {
            override fun getVerticalSnapPreference(): Int {
                return SNAP_TO_START
            }
        }

        // add scroll listener to detect when list is being dragged
        episodesList.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                super.onScrollStateChanged(recyclerView, newState)
                episodesListDragListener?.onEpisodesListDragStateChanged(newState)
            }
        })

        // set up edge to edge display
        setupEdgeToEdge()
    }


    /* Scrolls recycler list to top */
    fun scrollToTop() {
        scroller.targetPosition = 0
        layoutManager.startSmoothScroll(scroller)
    }


    /* Updates the podcast views */
    @SuppressLint("ClickableViewAccessibility")
    fun updatePodcastViews() {
        // set podcast name and cover (if they have not been set earlier)
        setPodcastName(podcast.name)
        setPodcastCover(podcast.cover)
        // set up podcast cover view
        coverView.contentDescription = "${rootView.context.getString(R.string.descr_player_podcast_cover)}: ${podcast.name}"
        coverView.transitionName = "podcast_cover_transition_${podcast.remotePodcastFeedLocation}"
        coverView.setOnLongClickListener {
            DownloadHelper.refreshCover(rootView.context, podcast)
            val v = rootView.context.getSystemService(Context.VIBRATOR_SERVICE) as Vibrator
            v.vibrate(50)
            // v.vibrate(VibrationEffect.createOneShot(50, android.os.VibrationEffect.DEFAULT_AMPLITUDE)); // todo check if there is an androidx vibrator
            return@setOnLongClickListener true
        }
        // set up podcast website view  (open website in browser)
        if (podcast.website.isNotEmpty()) {
            podcastWebsiteView.isVisible = true
            podcastLinksDivider.isVisible = true
            podcastWebsiteView
            podcastWebsiteView.setOnClickListener {
                rootView.context.startActivity(Intent(Intent.ACTION_VIEW, podcast.website.toUri()), null)
            }
        } else {
            podcastWebsiteView.isGone = true
            podcastLinksDivider.isGone = true
        }
        // podcast feed: set up clipboard copy
        podcastFeedView.setOnClickListener {
            val clip: ClipData = ClipData.newPlainText("simple text", podcast.remotePodcastFeedLocation)
            val cm: ClipboardManager = rootView.context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
            cm.setPrimaryClip(clip)
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.TIRAMISU){
                Toast.makeText(rootView.context, R.string.toast_message_copied_to_clipboard, Toast.LENGTH_LONG).show()
            }
        }
        // update the filter label
        updateFilterLabel(podcast.episodeListDisplayFilter)
    }


    /* Toggles the downloaded episodes filter */
    fun toggleFilter() {
        if (this::podcast.isInitialized) {
            when (podcast.episodeListDisplayFilter) {
                Keys.FILTER_SHOW_DOWNLOADED -> {
                    CoroutineScope(IO).launch {
                        collectionDatabase.podcastDao().updateEpisodeListFilter(podcast.remotePodcastFeedLocation, Keys.FILTER_SHOW_ALL)
                        withContext(Main) {
                            updateFilterLabel(Keys.FILTER_SHOW_ALL)
                            filterEpisodeList(Keys.FILTER_SHOW_ALL)
                        }
                    }
                }
                else -> {
                    CoroutineScope(IO).launch {
                        collectionDatabase.podcastDao().updateEpisodeListFilter(podcast.remotePodcastFeedLocation, Keys.FILTER_SHOW_DOWNLOADED)
                        withContext(Main) {
                            updateFilterLabel(Keys.FILTER_SHOW_DOWNLOADED)
                            filterEpisodeList(Keys.FILTER_SHOW_DOWNLOADED)
                        }
                    }
                }
            }
        }
    }


    /* Update the downloaded episodes filter label */
    private fun updateFilterLabel(filter: Int) {
        when (filter) {
            Keys.FILTER_SHOW_DOWNLOADED -> {
                filterText.text = rootView.context.getString(R.string.episode_list_filter_downloaded)
            }

            else -> {
                filterText.text = rootView.context.getString(R.string.episode_list_filter_all)
            }
        }
    }


    /* Filter the episode list */
    private fun filterEpisodeList(filter: Int) {
        val adapter: Filterable = episodesList.adapter as Filterable
        adapter.filter.filter(filter.toString())
    }


    /* Updates the podcast description view */
    fun updatePodcastDescriptionView() {
        // podcast description: fit into available space in layout / expands on tap
        // Credit: https://stackoverflow.com/a/46904071/14326132
        if (this::podcastDescription.isInitialized) {
            // set the description
            podcastDescriptionView.setCleanHtml(podcastDescription.description)
            // check if text fade needed
            podcastDescriptionView.viewTreeObserver.addOnPreDrawListener (object : OnPreDrawListener {
                override fun onPreDraw(): Boolean {
                    podcastDescriptionView.viewTreeObserver.removeOnPreDrawListener(this)
                    val numberOfLinesVisible: Int = podcastDescriptionView.height / podcastDescriptionView.lineHeight
                    podcastDescriptionView.setCleanHtml(podcastDescription.description)
                    val podcastDescriptionLineCount: Int = podcastDescriptionView.lineCount
                    podcastDescriptionView.maxLines = numberOfLinesVisible
                    togglePodcastDescriptionFadeView(podcastDescriptionLineCount > podcastDescriptionView.maxLines)
                    return true
                }
            })
            podcastDescriptionView.setOnClickListener {
                DescriptionDialog().show(rootView.context, podcast, podcastDescription.description)
            }
        }
    }


    /* Toggles the text fade at the end of the description */
    private fun togglePodcastDescriptionFadeView(podcastDescriptionTooLong: Boolean) {
        if (podcastDescriptionTooLong) {
            podcastDescriptionFadeView.visibility = View.VISIBLE
        } else {
            podcastDescriptionFadeView.visibility = View.GONE
        }
    }


    /* Sets the podcast name */
    fun setPodcastName(podcastName: String) {
        // only set, if it has not yet been set
        if (toolbar.title != podcastName) {
            toolbar.title = podcastName
        }
    }


    /* Sets the podcast cover */
    fun setPodcastCover(newPodcastCoverUriString: String) {
        // only set, if it has not yet been set
        if (podcastCoverUriString != newPodcastCoverUriString) {
            Glide.with(rootView.context)
                .load(newPodcastCoverUriString)
                .error(R.drawable.ic_default_cover_rss_icon_192dp)
                .into(coverView)
            podcastCoverUriString = newPodcastCoverUriString
        }
    }


    /* Sets the podcast cover */
    fun setPodcastCoverAndStartPostponedEnterTransition(activity: FragmentActivity, newPodcastCoverUriString: String) {
        if (podcastCoverUriString != newPodcastCoverUriString) {
            Glide.with(rootView.context)
                .load(newPodcastCoverUriString)
                .error(R.drawable.ic_default_cover_rss_icon_192dp)
                .dontAnimate() // avoid Glide's crossfade
                .listener(object : RequestListener<Drawable> {
                    override fun onLoadFailed(e: GlideException?, model: Any?, target: com.bumptech.glide.request.target.Target<Drawable?>?, isFirstResource: Boolean): Boolean {
                        startPostponedEnterTransition(activity)
                        return false // return false so error placeholder can be applied
                    }

                    override fun onResourceReady(resource: Drawable?, model: Any?, target: com.bumptech.glide.request.target.Target<Drawable?>?, dataSource: DataSource?, isFirstResource: Boolean): Boolean {
                        startPostponedEnterTransition(activity)
                        return false // return false so Glide can set the resource to the target.
                    }
                })
                .into(coverView)
        } else {
            startPostponedEnterTransition(activity)
        }
        podcastCoverUriString = newPodcastCoverUriString
    }


    /* Sets up margins/paddings for edge to edge view - for API 35 and above */
    private fun setupEdgeToEdge() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM) {
            ViewCompat.setOnApplyWindowInsetsListener(rootView) { v, insets ->
                // get measurements for status and navigation bar
                systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars() or WindowInsetsCompat.Type.displayCutout())

                // apply measurements
                toolbar.updateLayoutParams<ConstraintLayout.LayoutParams> {
                    topMargin = systemBars.top
                }
                episodesList.updatePadding (
                    bottom = systemBars.bottom + ((Keys.PLAYER_HEIGHT + Keys.PLAYER_BOTTOM_MARGIN) * UiHelper.getDensityScalingFactor(rootView.context)).toInt()
                )

                // return the insets
                insets
            }
        }
    }


    /*
     * Inner class: Custom LinearLayoutManager
     */
    private inner class CustomLayoutManager(context: Context): LinearLayoutManager(context, VERTICAL, false) {
        override fun supportsPredictiveItemAnimations(): Boolean {
            return true
        }
    }
    /*
     * End of inner class
     */
}