/*
 * Podcast.kt
 * Implements the Podcast class
 * A Podcast object holds the base data of a podcast
 *
 * This file is part of
 * ESCAPEPOD - Free and Open Podcast App
 *
 * Copyright (c) 2018-25 - Y20K.org
 * Licensed under the MIT-License
 * http://opensource.org/licenses/MIT
 */


package org.y20k.escapepod.database.objects

import androidx.room.ColumnInfo
import androidx.room.Entity
import androidx.room.Index
import androidx.room.PrimaryKey
import org.y20k.escapepod.Keys
import org.y20k.escapepod.xml.RssHelper
import java.util.Date

/*
 * Podcast class
 */
@Entity(tableName = "podcasts", indices = [Index(value = ["remote_podcast_feed_location"], unique = true), Index(value = ["remote_image_file_location"], unique = false), Index(value = ["latest_episode_date"], unique = false)])
data class Podcast(

        @PrimaryKey
        @ColumnInfo (name = "remote_podcast_feed_location") val remotePodcastFeedLocation: String,

        @ColumnInfo (name = "name") val name: String,
        @ColumnInfo (name = "website") val website: String,
        @ColumnInfo (name = "cover") val cover: String,
        @ColumnInfo (name = "small_cover") val smallCover: String,
        @ColumnInfo (name = "latest_episode_date") val latestEpisodeDate: Date,
        @ColumnInfo (name = "remote_image_file_location") val remoteImageFileLocation: String,
        @ColumnInfo (name = "auto_download_episodes") val autoDownloadEpisodes: Boolean,
        @ColumnInfo (name = "episode_list_display_filter") val episodeListDisplayFilter: Int

) {


    /* Constructor used when cover has been set */
    constructor(podcast: Podcast, cover: String, smallCover: String) : this (
        name = podcast.name,
        website = podcast.website,
        cover = cover,           // <= set cover
        smallCover = smallCover, // <= set small cover
        latestEpisodeDate = podcast.latestEpisodeDate,
        remoteImageFileLocation = podcast.remoteImageFileLocation,
        remotePodcastFeedLocation = podcast.remotePodcastFeedLocation,
        autoDownloadEpisodes = podcast.autoDownloadEpisodes,
        episodeListDisplayFilter = podcast.episodeListDisplayFilter
    )


    /* Constructor that uses output from RssHelper*/
    constructor(rssPodcast: RssHelper.RssPodcast) : this (
        name = rssPodcast.name,
        website = rssPodcast.website,
        cover = rssPodcast.cover,
        smallCover = rssPodcast.smallCover,
        latestEpisodeDate = rssPodcast.latestEpisodeDate,
        remoteImageFileLocation = rssPodcast.remoteImageFileLocation,
        remotePodcastFeedLocation = rssPodcast.remotePodcastFeedLocation,
        autoDownloadEpisodes = Keys.DEFAULT_PODCAST_AUTO_DOWNLOAD_EPISODES,
        episodeListDisplayFilter = Keys.FILTER_SHOW_ALL
    )


//    /* Creates MediaItem for a podcast - used by collection provider */
//    fun toMediaMetaItem(): MediaBrowserCompat.MediaItem {
//        val mediaDescriptionBuilder = MediaDescriptionCompat.Builder()
//        mediaDescriptionBuilder.setTitle(name)
//        mediaDescriptionBuilder.setIconUri(cover.toUri())
//        return MediaBrowserCompat.MediaItem(mediaDescriptionBuilder.build(), MediaBrowserCompat.MediaItem.FLAG_BROWSABLE)
//    }

}