// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#define ENABLE_DIALOG_LOOK

#include "Main.hpp"
#include "ui/window/SingleWindow.hpp"
#include "ui/canvas/Canvas.hpp"
#include "ui/control/List.hpp"
#include "Look/ChartLook.hpp"
#include "Form/Button.hpp"
#include "util/Macros.hpp"
#include "Renderer/ChartRenderer.hpp"

static const TCHAR *const chart_names[] = {
  _T("Line"),
  _T("Line2"),
};

class ChartWindow : public PaintWindow {
  unsigned chart;
  const ChartLook &look;

public:
  ChartWindow(const ChartLook &_look)
    :look(_look)
  {
    SetChart(0);
  }

  void SetChart(unsigned _chart) {
    assert(_chart < ARRAY_SIZE(chart_names));
    chart = _chart;
  }

protected:
  void OnPaint(Canvas &canvas) noexcept override;

  void DrawChart(ChartRenderer &renderer);
};

void
ChartWindow::OnPaint(Canvas &canvas) noexcept
{
  canvas.ClearWhite();
  ChartRenderer renderer(look, canvas, canvas.GetRect());
  DrawChart(renderer);
}

void
ChartWindow::DrawChart(ChartRenderer &renderer)
{
  if (chart == 1) {
    renderer.SetXLabel(_T("VVV"),_T("m/s"));
    renderer.SetYLabel(_T("AAA"),_T("m/s"));
  }

  renderer.Begin();

  renderer.ScaleXFromValue(0);
  renderer.ScaleXFromValue(100);

  renderer.ScaleYFromValue(0);
  renderer.ScaleYFromValue(100);

  if (chart == 0) {
    renderer.DrawLine({0, 10}, {100, 70},
                      look.GetPen(ChartLook::STYLE_BLUETHINDASH));
  } else if (chart == 1) {
    renderer.ScaleXFromValue(-50);
    renderer.ScaleXFromValue(110);
    renderer.ScaleYFromValue(110);

    renderer.DrawLine({0, 10}, {100, 70},
                      look.GetPen(ChartLook::STYLE_BLUETHINDASH));

    renderer.DrawLine({0, 10}, {100, 80},
                      look.GetPen(ChartLook::STYLE_GREENDASH));

    renderer.DrawLine({0, 10}, {100, 100},
                      look.GetPen(ChartLook::STYLE_BLACK));

    renderer.DrawXGrid(20, 20, ChartRenderer::UnitFormat::NUMERIC);

    renderer.DrawYGrid(20, 20, ChartRenderer::UnitFormat::NUMERIC);

    renderer.DrawLabel({50, 50}, _T("hello"));
  }

  renderer.Finish();
}

class TestWindow : public UI::SingleWindow,
                   ListItemRenderer, ListCursorHandler {
  Button close_button;
  ListControl *type_list = nullptr;
  ChartWindow chart;

public:
  TestWindow(UI::Display &display, const ChartLook &chart_look)
    :UI::SingleWindow(display), chart(chart_look) {}
  ~TestWindow() {
    delete type_list;
  }

  void Create(const DialogLook &look, PixelSize size) {
    SingleWindow::Create(_T("RunChartRenderer"), size);

    const PixelRect rc = GetClientRect();

    WindowStyle with_border;
    with_border.Border();

    const PixelRect list_rc(0, 0, 250, rc.bottom - 30);

    type_list = new ListControl(*this, look, list_rc,
                                with_border, 25);

    type_list->SetItemRenderer(this);
    type_list->SetCursorHandler(this);
    type_list->SetLength(ARRAY_SIZE(chart_names));

    PixelRect chart_rc = rc;
    chart_rc.left = list_rc.right;
    chart.Create(*this, chart_rc, with_border);

    PixelRect button_rc = rc;
    button_rc.right = list_rc.right;
    button_rc.top = button_rc.bottom - 30;
    close_button.Create(*this, *button_look, _T("Close"), button_rc,
                        WindowStyle(),
                        [this](){ Close(); });

    type_list->SetFocus();
  }

protected:
  /* virtual methods from ListItemRenderer */
  void OnPaintItem(Canvas &canvas, const PixelRect rc,
                   unsigned idx) noexcept override {
    assert(idx < ARRAY_SIZE(chart_names));

    canvas.DrawText(rc.WithPadding(2).GetTopLeft(), chart_names[idx]);
  }

  /* virtual methods from ListCursorHandler */
  void OnCursorMoved(unsigned idx) noexcept override {
    assert(idx < ARRAY_SIZE(chart_names));

    chart.SetChart(idx);
  }
};

static void
Main(UI::Display &display)
{
  ChartLook chart_look;
  chart_look.Initialise();

  TestWindow window{display, chart_look};
  window.Create(*dialog_look, {640, 480});

  window.Show();
  window.RunEventLoop();
}
