// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#include "Terrain/RasterMap.hpp"
#include "Geo/GeoClip.hpp"
#include "Math/Util.hpp"

#include <algorithm>
#include <cassert>

void
RasterMap::UpdateProjection() noexcept
{
  projection.Set(GetBounds(), raster_tile_cache.GetFineSize());
}

void
RasterMap::LoadCache(BufferedReader &r)
{
  raster_tile_cache.LoadCache(r);
  UpdateProjection();
}

TerrainHeight
RasterMap::GetHeight(const GeoPoint &location) const noexcept
{
  const auto pt = projection.ProjectCoarse(location);
  return raster_tile_cache.GetHeight(pt);
}

TerrainHeight
RasterMap::GetInterpolatedHeight(const GeoPoint &location) const noexcept
{
  const auto pt = projection.ProjectFine(location);
  return raster_tile_cache.GetInterpolatedHeight(pt);
}

void
RasterMap::ScanLine(const GeoPoint &start, const GeoPoint &end,
                    TerrainHeight *buffer, unsigned size,
                    bool interpolate) const noexcept
{
  assert(buffer != nullptr);
  assert(size > 0);

  constexpr TerrainHeight invalid = TerrainHeight::Invalid();

  const double total_distance = start.DistanceS(end);
  if (total_distance <= 0) {
    std::fill_n(buffer, size, invalid);
    return;
  }

  /* clip the line to the map bounds */

  GeoPoint clipped_start = start, clipped_end = end;
  const GeoClip clip(GetBounds());
  if (!clip.ClipLine(clipped_start, clipped_end)) {
    std::fill_n(buffer, size, invalid);
    return;
  }

  double clipped_start_distance =
    std::max(clipped_start.DistanceS(start), 0.);
  double clipped_end_distance =
    std::max(clipped_end.DistanceS(start), 0.);

  /* calculate the offsets of the clipped range within the buffer */

  unsigned clipped_start_offset =
    (unsigned)(size * clipped_start_distance / total_distance);
  unsigned clipped_end_offset =
    uround(size * clipped_end_distance / total_distance);
  if (clipped_end_offset > size)
    clipped_end_offset = size;
  if (clipped_start_offset + 2 > clipped_end_offset) {
    std::fill_n(buffer, size, invalid);
    return;
  }

  assert(clipped_start_offset < size);
  assert(clipped_end_offset <= size);

  /* fill the two regions which are outside the map  */

  std::fill(buffer, buffer + clipped_start_offset, invalid);
  std::fill(buffer + clipped_end_offset, buffer + size, invalid);

  /* now scan the middle part which is within the map */

  const auto fine_size = raster_tile_cache.GetFineSize();

  RasterLocation raster_start = projection.ProjectFine(clipped_start);
  if (raster_start.x >= fine_size.x)
    raster_start.x = fine_size.x - 1;
  if (raster_start.y >= fine_size.y)
    raster_start.y = fine_size.y - 1;

  RasterLocation raster_end = projection.ProjectFine(clipped_end);
  if (raster_end.x >= fine_size.x)
    raster_end.x = fine_size.x - 1;
  if (raster_end.y >= fine_size.y)
    raster_end.y = fine_size.y - 1;

  raster_tile_cache.ScanLine(raster_start, raster_end,
                             buffer + clipped_start_offset,
                             clipped_end_offset - clipped_start_offset,
                             interpolate);
}

RasterMap::Intersection
RasterMap::FirstIntersection(const GeoPoint &origin, const int h_origin,
                             const GeoPoint &destination, const int h_destination,
                             const int h_virt, const int h_ceiling,
                             const int h_safety) const noexcept
{
  const auto c_origin = projection.ProjectCoarseRound(origin);
  const auto c_destination = projection.ProjectCoarseRound(destination);
  const int c_diff = ManhattanDistance(c_origin, c_destination);
  const bool can_climb = (h_destination< h_virt);

  if (c_diff==0) {
    return Intersection::Invalid(); // no distance
  }

  const int slope_fact = (((int)h_virt) << RASTER_SLOPE_FACT) / c_diff;
  const int vh_origin = std::max(h_origin,
                                 h_destination
                                 - ((c_diff * slope_fact) >> RASTER_SLOPE_FACT));

  const auto intersection =
    raster_tile_cache.FirstIntersection(c_origin, c_destination,
                                        vh_origin, h_destination,
                                        slope_fact, h_ceiling, h_safety,
                                        can_climb);
  if (!intersection ||
      intersection->location == c_destination ||
      intersection->height <= h_destination)
    return Intersection::Invalid();

  return {projection.UnprojectCoarse(intersection->location), intersection->height};
}

GeoPoint
RasterMap::GroundIntersection(const GeoPoint &origin,
                              const int h_origin, const int h_glide,
                              const GeoPoint &destination,
                              const int height_floor) const noexcept
{
  const auto c_origin = projection.ProjectCoarseRound(origin);
  const auto c_destination = projection.ProjectCoarseRound(destination);
  const int c_diff = ManhattanDistance(c_origin, c_destination);
  if (c_diff == 0)
    return GeoPoint::Invalid();

  const int slope_fact = (((int)h_glide) << RASTER_SLOPE_FACT) / c_diff;

  auto c_int =
    raster_tile_cache.GroundIntersection(c_origin, c_destination,
                                         h_origin, slope_fact, height_floor);
  if (c_int.x < 0)
    return GeoPoint::Invalid();

  return projection.UnprojectCoarse(c_int);
}
