/*
 * Copyright 2020 The TensorFlow Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// Modifications by woheller69

package org.tensorflow.lite.examples.soundclassifier

import android.Manifest
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.media.AudioManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.util.DisplayMetrics
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.webkit.WebSettings
import android.widget.Toast
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.preference.PreferenceManager
import com.google.android.material.slider.LabelFormatter.LABEL_GONE
import org.tensorflow.lite.examples.soundclassifier.databinding.ActivityMainBinding
import org.woheller69.freeDroidWarn.FreeDroidWarn

class MainActivity : BaseActivity() {

  private lateinit var soundClassifier: SoundClassifier
  private lateinit var binding: ActivityMainBinding

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)
    binding = ActivityMainBinding.inflate(layoutInflater)
    setContentView(binding.root)

    //On Android 15, both BottomAppBar and BottomNavigationView automatically register WindowInsetsListeners that:
    //Add extra padding for gesture navigation areas
    //Increase the minimum height to accommodate system bars
    //Modify layout behavior to "avoid" system UI (but overdo it)
    //We do not need that, therefore set them "null"
    binding.bottomAppBar.setOnApplyWindowInsetsListener(null)
    binding.bottomNavigationView.setOnApplyWindowInsetsListener(null)

    val sharedPref = PreferenceManager.getDefaultSharedPreferences(this)

    //Set aspect ratio for webview and icon
    val width = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
      val windowMetrics = windowManager.currentWindowMetrics
      windowMetrics.bounds.width()
    } else {
      val displayMetrics = DisplayMetrics()
      windowManager.defaultDisplay.getMetrics(displayMetrics)
      displayMetrics.widthPixels
    }
    val paramsWebview: ViewGroup.LayoutParams = binding.webview.getLayoutParams() as ViewGroup.LayoutParams
    paramsWebview.height = (width / 1.8f).toInt()
    val paramsIcon: ViewGroup.LayoutParams = binding.icon.getLayoutParams() as ViewGroup.LayoutParams
    paramsIcon.height = (width / 1.8f).toInt()

    binding.rangeSlider.labelBehavior = LABEL_GONE

    soundClassifier = SoundClassifier(this, binding, SoundClassifier.Options())
    binding.gps.setText(getString(R.string.latitude)+": --.-- / " + getString(R.string.longitude) + ": --.--" )
    binding.webview.setWebViewClient(object : MlWebViewClient(this) {})
    binding.webview.settings.setDomStorageEnabled(true)
    binding.webview.settings.setJavaScriptEnabled(true)

    binding.fab.setOnClickListener {
      if (binding.progressHorizontal.isIndeterminate) {
        binding.progressHorizontal.setIndeterminate(false)
        binding.fab.setImageDrawable(ContextCompat.getDrawable(this, R.drawable.ic_record_24dp))
        if (binding.icon.visibility == View.VISIBLE && sharedPref.getBoolean("show_spectrogram", false)){
          binding.rangeSlider.visibility = View.VISIBLE
          binding.runRecognizerButton.visibility = View.VISIBLE
          binding.resetButton.visibility = View.VISIBLE
          binding.rangeSlider.values = mutableListOf(0.0f, 100.0f)
        }
      }
      else {
        binding.progressHorizontal.setIndeterminate(true)
        binding.fab.setImageDrawable(ContextCompat.getDrawable(this, R.drawable.ic_pause_24dp))
        binding.rangeSlider.visibility = View.GONE
        binding.runRecognizerButton.visibility = View.GONE
        binding.resetButton.visibility = View.GONE
      }
    }
    binding.bottomNavigationView.setOnItemSelectedListener { item ->
      when (item.itemId) {
        R.id.action_view -> {
          intent = Intent(this, ViewActivity::class.java)
          startActivity(intent)
        }
        R.id.action_bird_info -> {
          intent = Intent(this, BirdInfoActivity::class.java)
          startActivity(intent)
        }
        R.id.action_settings -> {
          intent = Intent(this, SettingsActivity::class.java)
          startActivity(intent)
        }
      }
      true
    }



    val metaModelInfluence = sharedPref.getFloat("meta_model_influence", 60.0f)
    binding.metaInfluenceSlider.value = metaModelInfluence
    binding.metaInfluenceSlider.addOnChangeListener { slider, value, fromUser ->
      val editor=sharedPref.edit()
      editor.putFloat("meta_model_influence", value)
      editor.apply()
    }
    FreeDroidWarn.showWarningOnUpgrade(this, BuildConfig.VERSION_CODE)
    if (GithubStar.shouldShowStarDialog(this)) GithubStar.starDialog(this, "https://github.com/woheller69/whoBIRD")

    requestPermissions()

  }

  override fun onResume() {
    super.onResume()
    val audioManager = getSystemService(Context.AUDIO_SERVICE) as AudioManager

    val sharedPref = PreferenceManager.getDefaultSharedPreferences(this)
    if (sharedPref.getBoolean("bluetooth", false)){
      if (ActivityCompat.checkSelfPermission(this, Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
          requestPermissions(arrayOf(Manifest.permission.BLUETOOTH_CONNECT), REQUEST_PERMISSIONS)
        }
      }
      audioManager.startBluetoothSco()
      audioManager.isBluetoothScoOn = true
    } else {
      audioManager.stopBluetoothSco()
      audioManager.isBluetoothScoOn = false
    }

    LocationHelper.requestLocation(this, soundClassifier)
    if (!checkLocationPermission()){
      Toast.makeText(this, this.resources.getString(R.string.error_location_permission), Toast.LENGTH_SHORT).show()
    }
    if (checkMicrophonePermission()){
      soundClassifier.start()
    } else {
      Toast.makeText(this, this.resources.getString(R.string.error_audio_permission), Toast.LENGTH_SHORT).show()
    }
    keepScreenOn(true)
  }

  override fun onPause() {
    super.onPause()
    LocationHelper.stopLocation(this)
    if (soundClassifier.isRecording) soundClassifier.stop()
    val audioManager = getSystemService(Context.AUDIO_SERVICE) as AudioManager
    audioManager.stopBluetoothSco()
    audioManager.isBluetoothScoOn = false
  }

  private fun checkMicrophonePermission(): Boolean {
    if (ContextCompat.checkSelfPermission(this, Manifest.permission.RECORD_AUDIO ) == PackageManager.PERMISSION_GRANTED) {
      return true
    } else {
      return false
    }
  }

  private fun checkLocationPermission(): Boolean {
    if (ContextCompat.checkSelfPermission(this, Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
      return true
    } else {
      return false
    }
  }

  private fun requestPermissions() {
    val perms = mutableListOf<String>()
    if (ContextCompat.checkSelfPermission(this, Manifest.permission.RECORD_AUDIO) != PackageManager.PERMISSION_GRANTED) {
      perms.add(Manifest.permission.RECORD_AUDIO)
    }
    if (ContextCompat.checkSelfPermission(this, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
      perms.add(Manifest.permission.ACCESS_COARSE_LOCATION)
      perms.add(Manifest.permission.ACCESS_FINE_LOCATION)
    }
    if (!perms.isEmpty()) requestPermissions(perms.toTypedArray(), REQUEST_PERMISSIONS)
  }

  private fun keepScreenOn(enable: Boolean) =
    if (enable) {
      window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    } else {
      window.clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

  companion object {
    const val REQUEST_PERMISSIONS = 1337
  }

  fun reload(view: View) {
    binding.webview.settings.setCacheMode(WebSettings.LOAD_DEFAULT)
    binding.webview.loadUrl(binding.webviewUrl.text.toString())
  }

  override fun onCreateOptionsMenu(menu: Menu): Boolean {
    val inflater = menuInflater
    inflater.inflate(R.menu.main, menu)
    return true
  }

  override fun onOptionsItemSelected(item: MenuItem): Boolean {
    when (item.itemId) {
      R.id.action_share_app -> {
        val intent = Intent(Intent.ACTION_SEND)
        val shareBody = "https://f-droid.org/packages/org.woheller69.whobird/"
        intent.setType("text/plain")
        intent.putExtra(Intent.EXTRA_TEXT, shareBody)
        startActivity(Intent.createChooser(intent, ""))
        return true
      }
      R.id.action_info -> {
        startActivity(Intent(Intent.ACTION_VIEW, Uri.parse("https://github.com/woheller69/whobird")))
        return true
      }
      else -> return super.onOptionsItemSelected(item)
    }
  }

  fun runRecognizer(view: View) {
    if (view == binding.resetButton) binding.rangeSlider.values = mutableListOf(0.0f, 100.0f)

    binding.text1.setText("")
    binding.text1.setBackgroundResource(0)
    binding.text2.setText("")
    binding.text2.setBackgroundResource(0)

    val buffer = soundClassifier.getInputBufferSnapshot()
    val N: Int = buffer.capacity()
    // Ensure minPercentage <= maxPercentage
    val currentValues = binding.rangeSlider.values
    val minPercentage = currentValues[0]
    val maxPercentage = currentValues[1]

    // Calculate index bounds
    val lowerIndex = Math.floor(minPercentage / 100.0 * N).toInt()
    var upperIndex = Math.floor(maxPercentage / 100.0 * N).toInt()
    upperIndex = Math.min(upperIndex, N) // Clamp to buffer size

    // Zero out values outside the range
    for (i in 0 until N) {
      if (i < lowerIndex || i >= upperIndex) {
        buffer.put(i, 0.0f)
      }
    }
  soundClassifier.recognizeAndDisplay(buffer)
  }

}
