/* Copyright 2014 Eddy Xiao <bewantbe@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.woheller69.audio_analyzer_for_android;

import android.util.Log;

// See also res/values/arrays.xml
class ColorMapArray {

    static int[] selectColorMap(String colorMapName) {
        switch (colorMapName.toLowerCase()) {
            case "hot":
                return hot;
            case "jet":
                return jet;
            case "gray":
                return gray;
            case "blackbody_uniform":
                return blackbody_uniform;
            case "parula":
                return parula;
            case "magma":
                return magma;
            case "inferno":
                return inferno;
            case "plasma":
                return plasma;
            case "viridis":
                return viridis;
            default:
                Log.i("ColorMapArray:", "No this color map: " + colorMapName);
                return hot;
        }
    }

    // Generated by GNU Octave
    //   c=[0 0 0; hot(0xff)];
    //   v=int32(floor(flipud(c)*255.99)*[0x10000 0x100 1]');
    //   s=reshape(sprintf('0x%06x, ', v), 10*8, [])(1:end-1,:)';  s(end)=' '
    // Seems that Octave changed their hot map (to match Matlab?), this is the new map.
    static final int[] hot = {
        0xffffff, 0xfffffc, 0xfffff8, 0xfffff4, 0xfffff0, 0xffffec, 0xffffe8, 0xffffe4,
        0xffffe0, 0xffffdc, 0xffffd8, 0xffffd4, 0xffffd0, 0xffffcc, 0xffffc8, 0xffffc4,
        0xffffc0, 0xffffbd, 0xffffb9, 0xffffb5, 0xffffb1, 0xffffad, 0xffffa9, 0xffffa5,
        0xffffa1, 0xffff9d, 0xffff99, 0xffff95, 0xffff91, 0xffff8d, 0xffff89, 0xffff85,
        0xffff81, 0xffff7e, 0xffff7a, 0xffff76, 0xffff72, 0xffff6e, 0xffff6a, 0xffff66,
        0xffff62, 0xffff5e, 0xffff5a, 0xffff56, 0xffff52, 0xffff4e, 0xffff4a, 0xffff46,
        0xffff42, 0xffff3f, 0xffff3b, 0xffff37, 0xffff33, 0xffff2f, 0xffff2b, 0xffff27,
        0xffff23, 0xffff1f, 0xffff1b, 0xffff17, 0xffff13, 0xffff0f, 0xffff0b, 0xffff07,
        0xffff03, 0xffff00, 0xfffd00, 0xfffa00, 0xfff700, 0xfff500, 0xfff200, 0xffef00,
        0xffed00, 0xffea00, 0xffe700, 0xffe500, 0xffe200, 0xffdf00, 0xffdc00, 0xffda00,
        0xffd700, 0xffd400, 0xffd200, 0xffcf00, 0xffcc00, 0xffca00, 0xffc700, 0xffc400,
        0xffc200, 0xffbf00, 0xffbc00, 0xffb900, 0xffb700, 0xffb400, 0xffb100, 0xffaf00,
        0xffac00, 0xffa900, 0xffa700, 0xffa400, 0xffa100, 0xff9e00, 0xff9c00, 0xff9900,
        0xff9600, 0xff9400, 0xff9100, 0xff8e00, 0xff8c00, 0xff8900, 0xff8600, 0xff8400,
        0xff8100, 0xff7e00, 0xff7b00, 0xff7900, 0xff7600, 0xff7300, 0xff7100, 0xff6e00,
        0xff6b00, 0xff6900, 0xff6600, 0xff6300, 0xff6100, 0xff5e00, 0xff5b00, 0xff5800,
        0xff5600, 0xff5300, 0xff5000, 0xff4e00, 0xff4b00, 0xff4800, 0xff4600, 0xff4300,
        0xff4000, 0xff3d00, 0xff3b00, 0xff3800, 0xff3500, 0xff3300, 0xff3000, 0xff2d00,
        0xff2b00, 0xff2800, 0xff2500, 0xff2300, 0xff2000, 0xff1d00, 0xff1a00, 0xff1800,
        0xff1500, 0xff1200, 0xff1000, 0xff0d00, 0xff0a00, 0xff0800, 0xff0500, 0xff0200,
        0xff0000, 0xfd0000, 0xfa0000, 0xf70000, 0xf50000, 0xf20000, 0xef0000, 0xed0000,
        0xea0000, 0xe70000, 0xe50000, 0xe20000, 0xdf0000, 0xdc0000, 0xda0000, 0xd70000,
        0xd40000, 0xd20000, 0xcf0000, 0xcc0000, 0xca0000, 0xc70000, 0xc40000, 0xc20000,
        0xbf0000, 0xbc0000, 0xb90000, 0xb70000, 0xb40000, 0xb10000, 0xaf0000, 0xac0000,
        0xa90000, 0xa70000, 0xa40000, 0xa10000, 0x9e0000, 0x9c0000, 0x990000, 0x960000,
        0x940000, 0x910000, 0x8e0000, 0x8c0000, 0x890000, 0x860000, 0x840000, 0x810000,
        0x7e0000, 0x7b0000, 0x790000, 0x760000, 0x730000, 0x710000, 0x6e0000, 0x6b0000,
        0x690000, 0x660000, 0x630000, 0x610000, 0x5e0000, 0x5b0000, 0x580000, 0x560000,
        0x530000, 0x500000, 0x4e0000, 0x4b0000, 0x480000, 0x460000, 0x430000, 0x400000,
        0x3d0000, 0x3b0000, 0x380000, 0x350000, 0x330000, 0x300000, 0x2d0000, 0x2b0000,
        0x280000, 0x250000, 0x230000, 0x200000, 0x1d0000, 0x1a0000, 0x180000, 0x150000,
        0x120000, 0x100000, 0x0d0000, 0x0a0000, 0x080000, 0x050000, 0x020000, 0x000000
    };

    // c=[0 0 0; jet(0xff)];
    static final int[] jet = {
        0x830000, 0x870000, 0x8b0000, 0x8f0000, 0x930000, 0x970000, 0x9b0000, 0x9f0000,
        0xa30000, 0xa70000, 0xab0000, 0xaf0000, 0xb30000, 0xb70000, 0xbb0000, 0xbf0000,
        0xc30000, 0xc70000, 0xcb0000, 0xcf0000, 0xd30000, 0xd70000, 0xdb0000, 0xdf0000,
        0xe30000, 0xe70000, 0xeb0000, 0xef0000, 0xf30000, 0xf70000, 0xfb0000, 0xff0000,
        0xff0300, 0xff0700, 0xff0b00, 0xff0f00, 0xff1300, 0xff1700, 0xff1b00, 0xff1f00,
        0xff2300, 0xff2700, 0xff2b00, 0xff2f00, 0xff3300, 0xff3700, 0xff3b00, 0xff3f00,
        0xff4300, 0xff4700, 0xff4b00, 0xff4f00, 0xff5300, 0xff5700, 0xff5b00, 0xff5f00,
        0xff6300, 0xff6700, 0xff6b00, 0xff6f00, 0xff7300, 0xff7700, 0xff7b00, 0xff7f00,
        0xff8300, 0xff8700, 0xff8b00, 0xff8f00, 0xff9300, 0xff9700, 0xff9b00, 0xff9f00,
        0xffa300, 0xffa700, 0xffab00, 0xffaf00, 0xffb300, 0xffb700, 0xffbb00, 0xffbf00,
        0xffc300, 0xffc700, 0xffcb00, 0xffcf00, 0xffd300, 0xffd700, 0xffdb00, 0xffdf00,
        0xffe300, 0xffe700, 0xffeb00, 0xffef00, 0xfff300, 0xfff700, 0xfffb00, 0xffff00,
        0xfbff03, 0xf7ff07, 0xf3ff0b, 0xefff0f, 0xebff13, 0xe7ff17, 0xe3ff1b, 0xdfff1f,
        0xdbff23, 0xd7ff27, 0xd3ff2b, 0xcfff2f, 0xcbff33, 0xc7ff37, 0xc3ff3b, 0xbfff3f,
        0xbbff43, 0xb7ff47, 0xb3ff4b, 0xafff4f, 0xabff53, 0xa7ff57, 0xa3ff5b, 0x9fff5f,
        0x9bff63, 0x97ff67, 0x93ff6b, 0x8fff6f, 0x8bff73, 0x87ff77, 0x83ff7b, 0x7fff7f,
        0x7bff83, 0x77ff87, 0x73ff8b, 0x6fff8f, 0x6bff93, 0x67ff97, 0x63ff9b, 0x5fff9f,
        0x5bffa3, 0x57ffa7, 0x53ffab, 0x4fffaf, 0x4bffb3, 0x47ffb7, 0x43ffbb, 0x3fffbf,
        0x3bffc3, 0x37ffc7, 0x33ffcb, 0x2fffcf, 0x2bffd3, 0x27ffd7, 0x23ffdb, 0x1fffdf,
        0x1bffe3, 0x17ffe7, 0x13ffeb, 0x0fffef, 0x0bfff3, 0x07fff7, 0x03fffb, 0x00ffff,
        0x00fbff, 0x00f7ff, 0x00f3ff, 0x00efff, 0x00ebff, 0x00e7ff, 0x00e3ff, 0x00dfff,
        0x00dbff, 0x00d7ff, 0x00d3ff, 0x00cfff, 0x00cbff, 0x00c7ff, 0x00c3ff, 0x00bfff,
        0x00bbff, 0x00b7ff, 0x00b3ff, 0x00afff, 0x00abff, 0x00a7ff, 0x00a3ff, 0x009fff,
        0x009bff, 0x0097ff, 0x0093ff, 0x008fff, 0x008bff, 0x0087ff, 0x0083ff, 0x007fff,
        0x007bff, 0x0077ff, 0x0073ff, 0x006fff, 0x006bff, 0x0067ff, 0x0063ff, 0x005fff,
        0x005bff, 0x0057ff, 0x0053ff, 0x004fff, 0x004bff, 0x0047ff, 0x0043ff, 0x003fff,
        0x003bff, 0x0037ff, 0x0033ff, 0x002fff, 0x002bff, 0x0027ff, 0x0023ff, 0x001fff,
        0x001bff, 0x0017ff, 0x0013ff, 0x000fff, 0x000bff, 0x0007ff, 0x0003ff, 0x0000ff,
        0x0000fb, 0x0000f7, 0x0000f3, 0x0000ef, 0x0000eb, 0x0000e7, 0x0000e3, 0x0000df,
        0x0000db, 0x0000d7, 0x0000d3, 0x0000cf, 0x0000cb, 0x0000c7, 0x0000c3, 0x0000bf,
        0x0000bb, 0x0000b7, 0x0000b3, 0x0000af, 0x0000ab, 0x0000a7, 0x0000a3, 0x00009f,
        0x00009b, 0x000097, 0x000093, 0x00008f, 0x00008b, 0x000087, 0x000083, 0x000000
    };

    // c=[0 0 0; gray(0xff)];
    static final int[] gray = {
        0xffffff, 0xfefefe, 0xfdfdfd, 0xfcfcfc, 0xfbfbfb, 0xfafafa, 0xf9f9f9, 0xf8f8f8,
        0xf7f7f7, 0xf6f6f6, 0xf5f5f5, 0xf4f4f4, 0xf3f3f3, 0xf2f2f2, 0xf1f1f1, 0xf0f0f0,
        0xefefef, 0xeeeeee, 0xededed, 0xececec, 0xebebeb, 0xeaeaea, 0xe9e9e9, 0xe8e8e8,
        0xe7e7e7, 0xe6e6e6, 0xe5e5e5, 0xe4e4e4, 0xe3e3e3, 0xe2e2e2, 0xe1e1e1, 0xe0e0e0,
        0xdfdfdf, 0xdedede, 0xdddddd, 0xdcdcdc, 0xdbdbdb, 0xdadada, 0xd9d9d9, 0xd8d8d8,
        0xd7d7d7, 0xd6d6d6, 0xd5d5d5, 0xd4d4d4, 0xd3d3d3, 0xd2d2d2, 0xd1d1d1, 0xd0d0d0,
        0xcfcfcf, 0xcecece, 0xcdcdcd, 0xcccccc, 0xcbcbcb, 0xcacaca, 0xc9c9c9, 0xc8c8c8,
        0xc7c7c7, 0xc6c6c6, 0xc5c5c5, 0xc4c4c4, 0xc3c3c3, 0xc2c2c2, 0xc1c1c1, 0xc0c0c0,
        0xbfbfbf, 0xbebebe, 0xbdbdbd, 0xbcbcbc, 0xbbbbbb, 0xbababa, 0xb9b9b9, 0xb8b8b8,
        0xb7b7b7, 0xb6b6b6, 0xb5b5b5, 0xb4b4b4, 0xb3b3b3, 0xb2b2b2, 0xb1b1b1, 0xb0b0b0,
        0xafafaf, 0xaeaeae, 0xadadad, 0xacacac, 0xababab, 0xaaaaaa, 0xa9a9a9, 0xa8a8a8,
        0xa7a7a7, 0xa6a6a6, 0xa5a5a5, 0xa4a4a4, 0xa3a3a3, 0xa2a2a2, 0xa1a1a1, 0xa0a0a0,
        0x9f9f9f, 0x9e9e9e, 0x9d9d9d, 0x9c9c9c, 0x9b9b9b, 0x9a9a9a, 0x999999, 0x989898,
        0x979797, 0x969696, 0x959595, 0x949494, 0x939393, 0x929292, 0x919191, 0x909090,
        0x8f8f8f, 0x8e8e8e, 0x8d8d8d, 0x8c8c8c, 0x8b8b8b, 0x8a8a8a, 0x898989, 0x888888,
        0x878787, 0x868686, 0x858585, 0x848484, 0x838383, 0x828282, 0x818181, 0x7f7f7f,
        0x7e7e7e, 0x7d7d7d, 0x7c7c7c, 0x7b7b7b, 0x7a7a7a, 0x797979, 0x787878, 0x777777,
        0x767676, 0x757575, 0x747474, 0x737373, 0x727272, 0x717171, 0x707070, 0x6f6f6f,
        0x6e6e6e, 0x6d6d6d, 0x6c6c6c, 0x6b6b6b, 0x6a6a6a, 0x696969, 0x686868, 0x676767,
        0x666666, 0x656565, 0x646464, 0x636363, 0x626262, 0x616161, 0x606060, 0x5f5f5f,
        0x5e5e5e, 0x5d5d5d, 0x5c5c5c, 0x5b5b5b, 0x5a5a5a, 0x595959, 0x585858, 0x575757,
        0x565656, 0x555555, 0x545454, 0x535353, 0x525252, 0x515151, 0x505050, 0x4f4f4f,
        0x4e4e4e, 0x4d4d4d, 0x4c4c4c, 0x4b4b4b, 0x4a4a4a, 0x494949, 0x484848, 0x474747,
        0x464646, 0x454545, 0x444444, 0x434343, 0x424242, 0x414141, 0x404040, 0x3f3f3f,
        0x3e3e3e, 0x3d3d3d, 0x3c3c3c, 0x3b3b3b, 0x3a3a3a, 0x393939, 0x383838, 0x373737,
        0x363636, 0x353535, 0x343434, 0x333333, 0x323232, 0x313131, 0x303030, 0x2f2f2f,
        0x2e2e2e, 0x2d2d2d, 0x2c2c2c, 0x2b2b2b, 0x2a2a2a, 0x292929, 0x282828, 0x272727,
        0x262626, 0x252525, 0x242424, 0x232323, 0x222222, 0x212121, 0x202020, 0x1f1f1f,
        0x1e1e1e, 0x1d1d1d, 0x1c1c1c, 0x1b1b1b, 0x1a1a1a, 0x191919, 0x181818, 0x171717,
        0x161616, 0x151515, 0x141414, 0x131313, 0x121212, 0x111111, 0x101010, 0x0f0f0f,
        0x0e0e0e, 0x0d0d0d, 0x0c0c0c, 0x0b0b0b, 0x0a0a0a, 0x090909, 0x080808, 0x070707,
        0x060606, 0x050505, 0x040404, 0x030303, 0x020202, 0x010101, 0x000000, 0x000000
    };

    // Generated by Octave script util/blackbody.m and cm_list.py
    // With help of https://github.com/bewantbe/colormap_uniformize
    static final int[] blackbody_uniform = {
            0xd2e2ff, 0xd4e1ff, 0xd5e1ff, 0xd6e0ff, 0xd7e0ff, 0xd8dffd, 0xd8dffb, 0xd9dff9,
            0xd9dff7, 0xd9def5, 0xdadef4, 0xdadef2, 0xdbdef0, 0xdbddee, 0xdbdded, 0xdcddeb,
            0xdcdcea, 0xdddce8, 0xdddce7, 0xdddce5, 0xdedbe4, 0xdedbe3, 0xdedbe1, 0xdfdbe0,
            0xdfdadf, 0xdfdade, 0xdfdadd, 0xe0dadc, 0xe0dada, 0xe0d9d9, 0xe1d9d8, 0xe1d9d6,
            0xe1d8d4, 0xe2d8d2, 0xe2d7d0, 0xe3d7ce, 0xe3d6cc, 0xe4d6c9, 0xe4d5c7, 0xe5d5c5,
            0xe5d4c3, 0xe6d4c0, 0xe6d3be, 0xe7d2bb, 0xe8d2b9, 0xe8d1b6, 0xe9d0b4, 0xe9cfb1,
            0xeaceae, 0xeaceac, 0xebcda9, 0xebcca6, 0xeccba4, 0xeccaa1, 0xedc99e, 0xedc89b,
            0xeec798, 0xeec695, 0xefc592, 0xefc48f, 0xefc38d, 0xf0c28a, 0xf0c087, 0xf1bf84,
            0xf1be81, 0xf1bd7d, 0xf2bb7a, 0xf2ba77, 0xf2b974, 0xf2b771, 0xf3b66e, 0xf3b46b,
            0xf3b368, 0xf3b165, 0xf3b062, 0xf3ae5f, 0xf3ad5c, 0xf4ab59, 0xf4a956, 0xf4a852,
            0xf3a64f, 0xf3a44c, 0xf3a349, 0xf3a146, 0xf39f43, 0xf39d40, 0xf39c3d, 0xf29a3a,
            0xf29837, 0xf29634, 0xf19530, 0xf1932d, 0xf1912a, 0xf08f27, 0xf08d23, 0xef8b20,
            0xef8a1c, 0xee8818, 0xee8614, 0xed840f, 0xed8209, 0xec8003, 0xec7e00, 0xeb7d00,
            0xeb7b00, 0xea7900, 0xe97700, 0xe97600, 0xe87400, 0xe77200, 0xe77000, 0xe66f00,
            0xe56d00, 0xe56b00, 0xe46a00, 0xe36800, 0xe26600, 0xe16400, 0xe16200, 0xe06100,
            0xdf5f00, 0xde5d00, 0xdd5b00, 0xdc5a00, 0xdc5800, 0xdb5600, 0xda5400, 0xd95200,
            0xd85100, 0xd74f00, 0xd64d00, 0xd54b00, 0xd44900, 0xd34800, 0xd24600, 0xd14400,
            0xd04200, 0xcf4000, 0xce3e00, 0xcd3c00, 0xcc3a00, 0xcb3800, 0xc93600, 0xc83400,
            0xc73200, 0xc63000, 0xc52e00, 0xc42c00, 0xc22a00, 0xc12700, 0xc02500, 0xbf2300,
            0xbd2000, 0xbc1e00, 0xbb1b00, 0xb91800, 0xb81600, 0xb71200, 0xb50f00, 0xb40b00,
            0xb20600, 0xb10200, 0xaf0000, 0xad0000, 0xab0000, 0xa90000, 0xa60000, 0xa40000,
            0xa20000, 0xa00000, 0x9e0000, 0x9c0000, 0x9a0000, 0x980000, 0x960000, 0x940000,
            0x910000, 0x8f0000, 0x8d0000, 0x8b0000, 0x890000, 0x870000, 0x850000, 0x830000,
            0x810000, 0x7f0000, 0x7d0000, 0x7b0000, 0x790000, 0x770000, 0x750000, 0x730000,
            0x710000, 0x6f0000, 0x6d0000, 0x6b0000, 0x690000, 0x670000, 0x650000, 0x630000,
            0x610000, 0x5f0000, 0x5d0000, 0x5b0000, 0x590000, 0x570000, 0x550000, 0x530000,
            0x510000, 0x4f0000, 0x4d0000, 0x4b0000, 0x4a0000, 0x480000, 0x460000, 0x440000,
            0x420000, 0x400000, 0x3e0000, 0x3c0000, 0x3a0000, 0x390000, 0x370000, 0x350000,
            0x330000, 0x310000, 0x2f0000, 0x2e0000, 0x2c0000, 0x2a0000, 0x280000, 0x260000,
            0x250000, 0x230000, 0x210000, 0x200000, 0x1e0000, 0x1c0000, 0x1b0000, 0x190000,
            0x170000, 0x160000, 0x140000, 0x130000, 0x110000, 0x100000, 0x0e0000, 0x0d0000,
            0x0b0000, 0x0a0000, 0x090000, 0x070000, 0x060000, 0x050000, 0x050000, 0x000000
    };

    // Matlab's new default colormap (since 2014b)
    // http://blogs.mathworks.com/steve/2014/10/13/a-new-colormap-for-matlab-part-1-introduction/
    // https://www.mathworks.com/content/dam/mathworks/tag-team/Objects/r/81137_92238v00_RainbowColorMap_57312.pdf
    // c=[0 0 0; parula(255)];
    // v=int32(floor(flipud(c)*255.99)*[2^16 2^8 1]');
    // s=reshape(sprintf('0x%06x, ', v), 10*8, []);
    // s=s(1:end-1,:)';  s(end)=' '
    static final int[] parula = {
        0xf9fb0d, 0xf9f90f, 0xf8f710, 0xf7f512, 0xf7f313, 0xf6f114, 0xf6ef15, 0xf5ed16,
        0xf5ec18, 0xf5ea19, 0xf5e81a, 0xf5e61b, 0xf5e51c, 0xf5e31e, 0xf5e21f, 0xf6e020,
        0xf6df21, 0xf6dd22, 0xf7dc23, 0xf7da24, 0xf8d925, 0xf8d726, 0xf9d627, 0xf9d528,
        0xfad329, 0xfbd22a, 0xfbd12b, 0xfcd02c, 0xfcce2d, 0xfdcd2e, 0xfdcc2f, 0xfeca30,
        0xfec932, 0xffc833, 0xffc634, 0xffc535, 0xffc437, 0xffc238, 0xffc139, 0xffc03b,
        0xfebf3d, 0xfdbe3e, 0xfcbd40, 0xfbbc42, 0xf9bb43, 0xf8ba45, 0xf6ba46, 0xf4ba48,
        0xf2b949, 0xf0b94a, 0xeeb94b, 0xecb94c, 0xeab94d, 0xe8b94f, 0xe6b950, 0xe4ba50,
        0xe2ba51, 0xe0ba52, 0xdeba53, 0xdcba54, 0xdaba55, 0xd8ba56, 0xd6bb57, 0xd4bb58,
        0xd2bb59, 0xd0bb59, 0xcebb5a, 0xccbc5b, 0xcabc5c, 0xc8bc5d, 0xc6bc5e, 0xc3bc5f,
        0xc1bc5f, 0xbfbd60, 0xbdbd61, 0xbbbd62, 0xb9bd63, 0xb7bd64, 0xb4bd65, 0xb2be66,
        0xb0be66, 0xaebe67, 0xabbe68, 0xa9be69, 0xa7be6a, 0xa4be6b, 0xa2bf6c, 0xa0bf6d,
        0x9dbf6e, 0x9bbf6f, 0x98bf70, 0x96bf71, 0x93bf72, 0x91bf73, 0x8ebf74, 0x8cbf75,
        0x89bf76, 0x87bf77, 0x84bf78, 0x81bf79, 0x7ebf7a, 0x7cbf7c, 0x79bf7d, 0x76bf7e,
        0x73bf7f, 0x70bf81, 0x6dbf82, 0x6abf83, 0x67bf85, 0x64be86, 0x61be87, 0x5ebe89,
        0x5bbe8a, 0x58be8c, 0x55bd8d, 0x52bd8f, 0x50bd90, 0x4dbc92, 0x4abc93, 0x47bc95,
        0x44bb96, 0x42bb98, 0x3fba9a, 0x3cba9b, 0x3aba9d, 0x37b99e, 0x35b9a0, 0x32b8a1,
        0x30b8a2, 0x2eb7a4, 0x2bb7a5, 0x29b6a7, 0x27b6a8, 0x25b5aa, 0x23b5ab, 0x21b4ac,
        0x1fb4ae, 0x1db3af, 0x1bb3b0, 0x19b2b2, 0x17b1b3, 0x15b1b4, 0x14b0b5, 0x12b0b7,
        0x10afb8, 0x0fafb9, 0x0daeba, 0x0cadbb, 0x0badbd, 0x0aacbe, 0x09acbf, 0x08abc0,
        0x07aac1, 0x06aac2, 0x06a9c3, 0x06a8c4, 0x05a8c5, 0x05a7c6, 0x05a6c7, 0x05a6c8,
        0x05a5c9, 0x05a4ca, 0x05a3cb, 0x05a2cb, 0x06a2cc, 0x06a1cd, 0x06a0ce, 0x069fce,
        0x069ecf, 0x069dd0, 0x079cd0, 0x079bd1, 0x079ad1, 0x0899d1, 0x0997d2, 0x0a96d2,
        0x0b95d2, 0x0c94d2, 0x0d92d2, 0x0e91d2, 0x0f90d2, 0x108fd2, 0x108dd2, 0x118cd3,
        0x128bd3, 0x128ad3, 0x1388d3, 0x1387d3, 0x1486d4, 0x1485d4, 0x1484d4, 0x1483d5,
        0x1482d5, 0x1481d6, 0x1380d6, 0x137fd7, 0x137ed7, 0x127dd8, 0x127cd8, 0x117bd9,
        0x117ada, 0x1079da, 0x0f78db, 0x0f77db, 0x0e76dc, 0x0d75dc, 0x0c75dd, 0x0b74dd,
        0x0a73de, 0x0972de, 0x0871df, 0x0670df, 0x056ee0, 0x046de0, 0x036ce1, 0x036be1,
        0x026ae1, 0x0169e1, 0x0168e2, 0x0167e2, 0x0165e2, 0x0264e1, 0x0362e1, 0x0561e1,
        0x085fe0, 0x0b5ddf, 0x105bdd, 0x1459dc, 0x1857d9, 0x1c54d6, 0x2052d3, 0x2450d0,
        0x274ecd, 0x294cca, 0x2c4ac6, 0x2d48c3, 0x2f46c0, 0x3145bd, 0x3243b9, 0x3341b6,
        0x3440b3, 0x343eb0, 0x353cad, 0x353ba9, 0x3639a6, 0x3638a3, 0x3636a0, 0x36359d,
        0x36339a, 0x363296, 0x363093, 0x352f90, 0x352d8d, 0x352c8a, 0x352a87, 0x000000
    };

    // color maps ('magma', 'inferno', 'plasma', 'viridis') from matplotlib in Python
    // https://bids.github.io/colormap/
    // See file util/cm_list.py for generating these color maps.

    static final int[] magma = {
            0xfcfdbf, 0xfcfbbd, 0xfcfabc, 0xfcf8ba, 0xfdf6b8, 0xfdf4b6, 0xfdf2b4, 0xfdf1b2,
            0xfdefb0, 0xfdedae, 0xfdebac, 0xfde9ab, 0xfde7a9, 0xfde6a7, 0xfde4a5, 0xfee2a3,
            0xfee0a2, 0xfedea0, 0xfedc9e, 0xfedb9c, 0xfed99b, 0xfed799, 0xfed597, 0xfed395,
            0xfed194, 0xfed092, 0xfece90, 0xffcc8f, 0xffca8d, 0xffc88c, 0xffc68a, 0xffc588,
            0xffc387, 0xffc185, 0xffbf84, 0xffbd82, 0xffbb81, 0xffb97f, 0xffb87e, 0xffb67c,
            0xffb47b, 0xffb27a, 0xffb078, 0xffae77, 0xffac76, 0xffab74, 0xffa973, 0xfea772,
            0xfea570, 0xfea36f, 0xfea16e, 0xfe9f6d, 0xfe9d6c, 0xfe9c6b, 0xfe9a6a, 0xfe9869,
            0xfd9668, 0xfd9467, 0xfd9266, 0xfd9065, 0xfd8e64, 0xfd8c63, 0xfc8b62, 0xfc8961,
            0xfc8761, 0xfc8560, 0xfb835f, 0xfb815f, 0xfb7f5e, 0xfa7d5e, 0xfa7b5d, 0xfa795d,
            0xf9785c, 0xf9765c, 0xf8745c, 0xf8725c, 0xf7705c, 0xf76e5c, 0xf66c5c, 0xf56b5c,
            0xf5695c, 0xf4675c, 0xf3655c, 0xf3645c, 0xf2625d, 0xf1605d, 0xf05f5d, 0xef5d5e,
            0xee5b5e, 0xee5a5f, 0xed585f, 0xeb5760, 0xea5561, 0xe95461, 0xe85362, 0xe75163,
            0xe65064, 0xe54f64, 0xe34e65, 0xe24c66, 0xe14b67, 0xdf4a67, 0xde4968, 0xdd4869,
            0xdb476a, 0xda466b, 0xd8456b, 0xd7446c, 0xd5436d, 0xd4436e, 0xd2426e, 0xd1416f,
            0xcf4070, 0xce3f71, 0xcc3f71, 0xca3e72, 0xc93d73, 0xc73d73, 0xc63c74, 0xc43b75,
            0xc23b75, 0xc13a76, 0xbf3976, 0xbd3977, 0xbc3878, 0xba3878, 0xb93779, 0xb73779,
            0xb5367a, 0xb4357a, 0xb2357b, 0xb0347b, 0xaf347b, 0xad337c, 0xab337c, 0xaa327d,
            0xa8327d, 0xa6317d, 0xa5317e, 0xa3307e, 0xa1307e, 0xa02f7f, 0x9e2f7f, 0x9d2e7f,
            0x9b2e7f, 0x992d80, 0x982c80, 0x962c80, 0x942b80, 0x932b80, 0x912a81, 0x902a81,
            0x8e2981, 0x8c2981, 0x8b2881, 0x892881, 0x882781, 0x862781, 0x842681, 0x832581,
            0x812581, 0x802482, 0x7e2482, 0x7c2382, 0x7b2382, 0x792282, 0x782181, 0x762181,
            0x752081, 0x732081, 0x711f81, 0x701e81, 0x6e1e81, 0x6d1d81, 0x6b1c81, 0x6a1c81,
            0x681b81, 0x671a80, 0x651a80, 0x631980, 0x621980, 0x601880, 0x5f177f, 0x5d177f,
            0x5c167f, 0x5a157e, 0x58157e, 0x57147e, 0x55147d, 0x54137d, 0x52127c, 0x51127c,
            0x4f117b, 0x4d117b, 0x4c107a, 0x4a1079, 0x481078, 0x470f78, 0x450f77, 0x430f76,
            0x420f75, 0x400f74, 0x3e0f72, 0x3d0f71, 0x3b0f70, 0x390f6e, 0x370f6c, 0x360f6b,
            0x341069, 0x321067, 0x311065, 0x2f1063, 0x2d1161, 0x2b115e, 0x2a115c, 0x28115a,
            0x271157, 0x251155, 0x241153, 0x221150, 0x21114e, 0x1f114b, 0x1e1149, 0x1c1046,
            0x1b1044, 0x1a1042, 0x190f3f, 0x170f3d, 0x160e3a, 0x150e38, 0x140d36, 0x130d33,
            0x110c31, 0x100c2f, 0x0f0b2d, 0x0e0a2a, 0x0d0a28, 0x0c0926, 0x0b0824, 0x0a0722,
            0x090720, 0x08061d, 0x07051b, 0x060519, 0x050417, 0x040415, 0x040313, 0x030311,
            0x02020f, 0x02020d, 0x01010b, 0x010109, 0x010007, 0x000006, 0x000004, 0x000000
    };

    // PS: Adobe Audition-like color map
    static final int[] inferno = {
            0xfdffa5, 0xfbfea1, 0xf9fd9d, 0xf8fc9a, 0xf7fb96, 0xf6f992, 0xf5f88e, 0xf4f78a,
            0xf3f586, 0xf3f482, 0xf2f37d, 0xf2f179, 0xf2ef75, 0xf2ee71, 0xf2ec6d, 0xf2ea69,
            0xf2e965, 0xf3e761, 0xf3e55d, 0xf4e359, 0xf4e156, 0xf4df52, 0xf5dd4f, 0xf5db4c,
            0xf6d949, 0xf6d845, 0xf7d642, 0xf7d43f, 0xf8d23d, 0xf8d03a, 0xf9ce37, 0xf9cc34,
            0xf9ca32, 0xfac82f, 0xfac62d, 0xfac42a, 0xfbc228, 0xfbc025, 0xfbbe23, 0xfbbc21,
            0xfcba1e, 0xfcb81c, 0xfcb61a, 0xfcb418, 0xfcb216, 0xfcb014, 0xfcae12, 0xfcac10,
            0xfcaa0e, 0xfca90d, 0xfca70b, 0xfca50a, 0xfca308, 0xfca107, 0xfc9f07, 0xfc9d06,
            0xfc9b06, 0xfc9906, 0xfb9806, 0xfb9606, 0xfb9406, 0xfb9207, 0xfa9008, 0xfa8e08,
            0xfa8d09, 0xf98b0b, 0xf9890c, 0xf8870d, 0xf8850e, 0xf88410, 0xf78211, 0xf78012,
            0xf67e14, 0xf67d15, 0xf57b16, 0xf47918, 0xf47719, 0xf3761a, 0xf3741c, 0xf2721d,
            0xf1711e, 0xf16f20, 0xf06e21, 0xef6c22, 0xee6a23, 0xed6925, 0xed6726, 0xec6627,
            0xeb6428, 0xea632a, 0xe9612b, 0xe8602c, 0xe75e2d, 0xe65d2f, 0xe55b30, 0xe45a31,
            0xe35932, 0xe25733, 0xe15635, 0xe05536, 0xdf5337, 0xde5238, 0xdd5139, 0xdc4f3b,
            0xdb4e3c, 0xd94d3d, 0xd84c3e, 0xd74a3f, 0xd64940, 0xd54841, 0xd34743, 0xd24644,
            0xd14545, 0xcf4446, 0xce4347, 0xcd4248, 0xcb4149, 0xca404a, 0xc93f4b, 0xc73e4c,
            0xc63d4d, 0xc43c4e, 0xc33b4f, 0xc23a50, 0xc03951, 0xbf3852, 0xbd3853, 0xbc3754,
            0xba3655, 0xb93556, 0xb73457, 0xb63458, 0xb43359, 0xb33259, 0xb1315a, 0xb0315b,
            0xae305c, 0xad2f5d, 0xab2f5d, 0xaa2e5e, 0xa82d5f, 0xa62d60, 0xa52c60, 0xa32b61,
            0xa22b62, 0xa02a62, 0x9f2a63, 0x9d2964, 0x9b2864, 0x9a2865, 0x982765, 0x972766,
            0x952667, 0x942567, 0x922568, 0x902468, 0x8f2468, 0x8d2369, 0x8c2369, 0x8a226a,
            0x88216a, 0x87216b, 0x85206b, 0x84206b, 0x821f6c, 0x801f6c, 0x7f1e6c, 0x7d1d6c,
            0x7c1d6d, 0x7a1c6d, 0x781c6d, 0x771b6d, 0x751b6e, 0x741a6e, 0x721a6e, 0x70196e,
            0x6f186e, 0x6d186e, 0x6c176e, 0x6a176e, 0x68166e, 0x67156e, 0x65156e, 0x64146e,
            0x62146e, 0x61136e, 0x5f126e, 0x5d126e, 0x5c116e, 0x5a116e, 0x59106e, 0x570f6d,
            0x550f6d, 0x540e6d, 0x520e6d, 0x500d6c, 0x4f0d6c, 0x4d0c6c, 0x4c0c6b, 0x4a0b6b,
            0x480b6a, 0x470a6a, 0x450a69, 0x430a68, 0x420968, 0x400967, 0x3e0966, 0x3c0965,
            0x3b0964, 0x390963, 0x370962, 0x360960, 0x34095f, 0x32095e, 0x300a5c, 0x2e0a5a,
            0x2d0a59, 0x2b0a57, 0x290b55, 0x270b53, 0x260b51, 0x240b4e, 0x220b4c, 0x210c4a,
            0x1f0c48, 0x1d0c45, 0x1c0c43, 0x1a0c40, 0x190b3e, 0x170b3b, 0x160b39, 0x140b37,
            0x130a34, 0x120a32, 0x10092f, 0x0f092d, 0x0e082b, 0x0d0828, 0x0c0726, 0x0a0724,
            0x090621, 0x08061f, 0x07051d, 0x06041b, 0x050418, 0x040316, 0x040314, 0x030212,
            0x020210, 0x02010e, 0x01010b, 0x010109, 0x010007, 0x000006, 0x000004, 0x000000
    };

    static final int[] plasma = {
            0xf0f921, 0xf1f724, 0xf1f625, 0xf2f426, 0xf2f226, 0xf3f127, 0xf4ef27, 0xf4ed27,
            0xf5eb26, 0xf6ea26, 0xf6e826, 0xf7e625, 0xf7e425, 0xf8e325, 0xf8e125, 0xf9df24,
            0xf9de24, 0xfadc24, 0xfada24, 0xfbd924, 0xfbd724, 0xfbd524, 0xfcd424, 0xfcd224,
            0xfcd024, 0xfdcf25, 0xfdcd25, 0xfdcb25, 0xfdca26, 0xfdc826, 0xfec727, 0xfec527,
            0xfec328, 0xfec228, 0xfec029, 0xfebf29, 0xfebd2a, 0xfebc2b, 0xfeba2b, 0xfeb92c,
            0xfeb72d, 0xfeb52e, 0xfeb42e, 0xfeb22f, 0xfeb130, 0xfeaf31, 0xfdae31, 0xfdac32,
            0xfdab33, 0xfda934, 0xfda835, 0xfca636, 0xfca536, 0xfca437, 0xfca238, 0xfba139,
            0xfb9f3a, 0xfb9e3b, 0xfb9c3c, 0xfa9b3c, 0xfa9a3d, 0xf9983e, 0xf9973f, 0xf99540,
            0xf89441, 0xf89342, 0xf79142, 0xf79043, 0xf78f44, 0xf68d45, 0xf68c46, 0xf58b47,
            0xf58948, 0xf48849, 0xf48749, 0xf3854a, 0xf2844b, 0xf2834c, 0xf1814d, 0xf1804e,
            0xf07f4f, 0xf07e50, 0xef7c50, 0xee7b51, 0xee7a52, 0xed7953, 0xed7754, 0xec7655,
            0xeb7556, 0xeb7456, 0xea7257, 0xe97158, 0xe87059, 0xe86f5a, 0xe76e5b, 0xe66c5c,
            0xe66b5c, 0xe56a5d, 0xe4695e, 0xe3685f, 0xe36660, 0xe26561, 0xe16462, 0xe06362,
            0xe06263, 0xdf6064, 0xde5f65, 0xdd5e66, 0xdc5d67, 0xdc5c68, 0xdb5b69, 0xda5969,
            0xd9586a, 0xd8576b, 0xd7566c, 0xd7556d, 0xd6546e, 0xd5536f, 0xd45170, 0xd35070,
            0xd24f71, 0xd14e72, 0xd04d73, 0xd04c74, 0xcf4b75, 0xce4976, 0xcd4877, 0xcc4778,
            0xcb4679, 0xca457a, 0xc9447a, 0xc8437b, 0xc7427c, 0xc6407d, 0xc53f7e, 0xc43e7f,
            0xc33d80, 0xc23c81, 0xc13b82, 0xc03a83, 0xbf3984, 0xbe3785, 0xbd3686, 0xbc3587,
            0xbb3488, 0xba3389, 0xb93289, 0xb8318a, 0xb72f8b, 0xb52e8c, 0xb42d8d, 0xb32c8e,
            0xb22b8f, 0xb12a90, 0xb02991, 0xaf2892, 0xad2693, 0xac2594, 0xab2495, 0xaa2395,
            0xa92296, 0xa72197, 0xa62098, 0xa51e99, 0xa41d9a, 0xa21c9a, 0xa11b9b, 0xa01a9c,
            0x9e199d, 0x9d189d, 0x9c179e, 0x9a159f, 0x9914a0, 0x9813a0, 0x9612a1, 0x9511a1,
            0x9410a2, 0x920fa3, 0x910ea3, 0x900da4, 0x8e0ca4, 0x8d0ba5, 0x8b09a5, 0x8a08a6,
            0x8807a6, 0x8707a6, 0x8606a7, 0x8405a7, 0x8304a7, 0x8104a7, 0x8003a8, 0x7e03a8,
            0x7d02a8, 0x7b02a8, 0x7a01a8, 0x7801a8, 0x7701a8, 0x7500a9, 0x7400a9, 0x7200a9,
            0x7100a8, 0x6f00a8, 0x6e00a8, 0x6c00a8, 0x6a00a8, 0x6900a8, 0x6700a8, 0x6600a7,
            0x6400a7, 0x6300a7, 0x6100a7, 0x5f00a6, 0x5e00a6, 0x5c00a6, 0x5b00a5, 0x5901a5,
            0x5801a5, 0x5601a4, 0x5401a4, 0x5301a3, 0x5101a3, 0x5002a2, 0x4e02a2, 0x4c02a1,
            0x4b02a1, 0x4902a0, 0x4702a0, 0x46039f, 0x44039e, 0x42039e, 0x41039d, 0x3f039c,
            0x3d039c, 0x3c039b, 0x3a049a, 0x38049a, 0x360499, 0x350498, 0x330497, 0x310497,
            0x2f0496, 0x2d0495, 0x2b0594, 0x2a0593, 0x280592, 0x260592, 0x230591, 0x210590,
            0x1f058f, 0x1d068e, 0x1b068d, 0x18068c, 0x16068a, 0x130689, 0x100788, 0x000000
    };

    static final int[] viridis = {
            0xfee724, 0xfbe723, 0xf9e721, 0xf7e61f, 0xf4e61e, 0xf2e61c, 0xefe61b, 0xede51a,
            0xeae519, 0xe8e519, 0xe5e418, 0xe2e418, 0xe0e418, 0xdde318, 0xdbe318, 0xd8e319,
            0xd5e21a, 0xd3e21b, 0xd0e21c, 0xcde11d, 0xcbe11e, 0xc8e120, 0xc5e021, 0xc3e023,
            0xc0df24, 0xbddf26, 0xbbdf27, 0xb8de29, 0xb5de2b, 0xb3dd2d, 0xb0dd2e, 0xaddd30,
            0xabdc32, 0xa8dc33, 0xa5db35, 0xa3db37, 0xa0da39, 0x9dda3a, 0x9bd93c, 0x98d93e,
            0x95d83f, 0x93d841, 0x90d743, 0x8ed744, 0x8bd646, 0x89d548, 0x86d549, 0x84d44b,
            0x81d44c, 0x7fd34e, 0x7cd24f, 0x7ad251, 0x77d152, 0x75d054, 0x73d055, 0x70cf57,
            0x6ece58, 0x6cce5a, 0x69cd5b, 0x67cc5c, 0x65cc5e, 0x62cb5f, 0x60ca60, 0x5ec962,
            0x5cc963, 0x5ac864, 0x58c765, 0x55c666, 0x53c668, 0x51c569, 0x4fc46a, 0x4dc36b,
            0x4bc26c, 0x49c26d, 0x48c16e, 0x46c06f, 0x44bf70, 0x42be71, 0x40bd72, 0x3ebd73,
            0x3dbc74, 0x3bbb75, 0x39ba76, 0x38b977, 0x36b878, 0x35b779, 0x33b779, 0x32b67a,
            0x30b57b, 0x2fb47c, 0x2eb37c, 0x2cb27d, 0x2bb17e, 0x2ab07f, 0x29af7f, 0x28af80,
            0x27ae81, 0x26ad81, 0x25ac82, 0x24ab82, 0x23aa83, 0x23a983, 0x22a884, 0x21a785,
            0x21a685, 0x20a586, 0x20a486, 0x1fa386, 0x1fa387, 0x1fa287, 0x1fa188, 0x1ea088,
            0x1e9f88, 0x1e9e89, 0x1e9d89, 0x1e9c89, 0x1e9b8a, 0x1e9a8a, 0x1e998a, 0x1e988b,
            0x1f978b, 0x1f968b, 0x1f958b, 0x1f948c, 0x1f938c, 0x20938c, 0x20928c, 0x20918c,
            0x20908d, 0x218f8d, 0x218e8d, 0x218d8d, 0x228c8d, 0x228b8d, 0x228a8d, 0x23898e,
            0x23888e, 0x23878e, 0x24868e, 0x24858e, 0x25848e, 0x25838e, 0x25828e, 0x26828e,
            0x26818e, 0x26808e, 0x277f8e, 0x277e8e, 0x287d8e, 0x287c8e, 0x287b8e, 0x297a8e,
            0x29798e, 0x29788e, 0x2a778e, 0x2a768e, 0x2b758e, 0x2b748e, 0x2b738e, 0x2c728e,
            0x2c718e, 0x2d708e, 0x2d708e, 0x2d6f8e, 0x2e6e8e, 0x2e6d8e, 0x2f6c8e, 0x2f6b8e,
            0x2f6a8e, 0x30698e, 0x30688e, 0x31678e, 0x31668e, 0x32658e, 0x32648e, 0x33638d,
            0x33628d, 0x33618d, 0x34608d, 0x345f8d, 0x355e8d, 0x355d8d, 0x365c8d, 0x365b8d,
            0x37598c, 0x37588c, 0x38578c, 0x38568c, 0x39558c, 0x39548c, 0x3a538b, 0x3a528b,
            0x3b518b, 0x3b508b, 0x3c4f8a, 0x3c4e8a, 0x3d4d8a, 0x3d4b8a, 0x3e4a89, 0x3e4989,
            0x3f4889, 0x3f4788, 0x404688, 0x404588, 0x414387, 0x414287, 0x414186, 0x424086,
            0x423f85, 0x433e85, 0x433c84, 0x433b84, 0x443a83, 0x443983, 0x443882, 0x453681,
            0x453581, 0x453480, 0x46337f, 0x46317e, 0x46307e, 0x462f7d, 0x472e7c, 0x472c7b,
            0x472b7a, 0x472a79, 0x472979, 0x482778, 0x482677, 0x482576, 0x482475, 0x482274,
            0x482173, 0x482071, 0x481e70, 0x481d6f, 0x481c6e, 0x481a6d, 0x48196c, 0x48186a,
            0x481669, 0x481568, 0x471466, 0x471265, 0x471164, 0x470f62, 0x470e61, 0x460c60,
            0x460b5e, 0x46095d, 0x46085b, 0x45065a, 0x450558, 0x450357, 0x440255, 0x000000
    };
}
