package org.wikipedia.util

import android.content.Context
import android.icu.text.RelativeDateTimeFormatter
import android.os.Build
import android.text.format.DateFormat
import org.wikipedia.R
import org.wikipedia.WikipediaApp
import org.wikipedia.extensions.getResources
import org.wikipedia.extensions.getString
import org.wikipedia.extensions.toLocalDate
import org.wikipedia.feed.model.UtcDate
import java.text.SimpleDateFormat
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.ZoneOffset
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.time.format.TextStyle
import java.time.temporal.ChronoUnit
import java.time.temporal.TemporalAccessor
import java.util.Calendar
import java.util.Date
import java.util.GregorianCalendar
import java.util.Locale
import java.util.TimeZone
import java.util.concurrent.ConcurrentHashMap

object DateUtil {
    private val DATE_FORMATS = ConcurrentHashMap<String, SimpleDateFormat>()
    private val DATE_TIME_FORMATTERS = ConcurrentHashMap<String, DateTimeFormatter>()

    // TODO: Switch to DateTimeFormatter when minSdk = 26.
    fun iso8601DateParse(date: String): Date {
        return Date.from(Instant.parse(date))
    }

    fun iso8601LocalDateTimeParse(timestamp: String): LocalDateTime {
        return LocalDateTime.ofInstant(Instant.parse(timestamp), ZoneId.systemDefault())
    }

    fun dbDateFormat(date: Date): String {
        return getCachedDateFormat("yyyyMMddHHmmss", Locale.ROOT, true).format(date)
    }

    fun dbDateParse(date: String): Date {
        return getCachedDateFormat("yyyyMMddHHmmss", Locale.ROOT, true).parse(date)!!
    }

    fun getFeedCardDateString(age: Int): String {
        return getShortDateString(UtcDate(age).baseCalendar.time)
    }

    fun getFeedCardShortDateString(date: Calendar): String {
        return getExtraShortDateString(date.time)
    }

    fun getMDYDateString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "MM/dd/yyyy")
    }

    fun getMonthOnlyDateString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "MMMM d")
    }

    fun getMonthOnlyDateStringFromTimeString(dateStr: String): String {
        return getMonthOnlyDateString(SimpleDateFormat("yyyy-MM-dd", Locale.getDefault()).parse(dateStr)!!)
    }

    fun getMonthOnlyDateString(date: LocalDate): String {
        return getDateStringWithSkeletonPattern(date, "MMMM d")
    }

    fun getMonthOnlyWithoutDayDateString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "MMMM")
    }

    fun getYearOnlyDateString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "yyyy")
    }

    fun getYMDDateString(date: LocalDate): String {
        return getCachedDateTimeFormatter("yyyyMMdd", Locale.getDefault(), utc = false, skeleton = false)
            .format(date)
    }

    fun getMMMMdYYYY(date: Date, utc: Boolean = true): String {
        return getCachedDateFormat("MMMM d, yyyy", Locale.getDefault(), utc).format(date)
    }

    private fun getExtraShortDateString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "MMM d")
    }

    fun getTimeString(context: Context, date: Date): String {
        val datePattern = if (DateFormat.is24HourFormat(context)) "HH:mm" else "hh:mm a"
        return getDateStringWithSkeletonPattern(date, datePattern)
    }

    fun getTimeString(context: Context, localDateTime: LocalDateTime): String {
        val datePattern = if (DateFormat.is24HourFormat(context)) "HH:mm" else "hh:mm a"
        return getDateStringWithSkeletonPattern(localDateTime, datePattern)
    }

    fun getShortDayWithTimeString(dateStr: String): String {
        return getDateStringWithSkeletonPattern(iso8601DateParse(dateStr), "MMM d HH:mm")
    }

    fun getShortDayWithTimeString(date: Date): String {
        return getDateStringWithSkeletonPattern(date, "MM/dd/yyyy HH:mm")
    }

    fun getTimeAndDateString(context: Context, date: Date): String {
        val datePattern = if (DateFormat.is24HourFormat(context)) "HH:mm, MMM d, yyyy" else "hh:mm a, MMM d, yyyy"
        return getDateStringWithSkeletonPattern(date, datePattern)
    }

    fun getTimeAndDateString(context: Context, dateStr: String): String {
        val datePattern = if (DateFormat.is24HourFormat(context)) "HH:mm, MMM d, yyyy" else "hh:mm a, MMM d, yyyy"
        return getDateStringWithSkeletonPattern(iso8601DateParse(dateStr), datePattern)
    }

    fun getDateAndTime(context: Context, date: Date): String {
        val datePattern = if (DateFormat.is24HourFormat(context)) "MMM d, yyyy, HH:mm" else "MMM d, yyyy, hh:mm a"
        return getCachedDateFormat(datePattern, Locale.getDefault(), false).format(date)
    }

    private fun getDateStringWithSkeletonPattern(date: Date, pattern: String): String {
        return getCachedDateFormat(pattern, Locale.getDefault(), utc = false, skeleton = true)
            .format(date)
    }

    private fun getDateStringWithSkeletonPattern(temporalAccessor: TemporalAccessor, pattern: String): String {
        return getCachedDateTimeFormatter(pattern, Locale.getDefault(), utc = false, skeleton = true)
            .format(temporalAccessor)
    }

    private fun getCachedDateFormat(pattern: String, locale: Locale, utc: Boolean, skeleton: Boolean = false): SimpleDateFormat {
        return DATE_FORMATS.getOrPut(pattern) {
            val df = SimpleDateFormat(if (skeleton) DateFormat.getBestDateTimePattern(locale, pattern) else pattern, locale)
            if (utc) {
                df.timeZone = TimeZone.getTimeZone("UTC")
            }
            df
        }
    }

    private fun getCachedDateTimeFormatter(pattern: String, locale: Locale, utc: Boolean, skeleton: Boolean = false): DateTimeFormatter {
        return DATE_TIME_FORMATTERS.getOrPut(pattern) {
            val dtf = DateTimeFormatter.ofPattern(if (skeleton) DateFormat.getBestDateTimePattern(locale, pattern) else pattern, locale)
            return if (utc) dtf.withZone(ZoneOffset.UTC) else dtf
        }
    }

    fun getShortDateString(date: Date): String {
        // todo: consider allowing TWN date formats. It would be useful to have but might be
        //       difficult for translators to write correct format specifiers without being able to
        //       test them. We should investigate localization support in date libraries such as
        //       Joda-Time and how TWN solves this classic problem.
        val dateFormat = DateFormat.getMediumDateFormat(WikipediaApp.instance)
        dateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return dateFormat.format(date)
    }

    fun getShortDateString(localDate: LocalDate): String {
        return DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM).format(localDate)
    }

    fun getUtcRequestDateFor(age: Int): UtcDate {
        return UtcDate(age)
    }

    fun getDefaultDateFor(age: Int): Calendar {
        val calendar = Calendar.getInstance(TimeZone.getDefault())
        calendar.add(Calendar.DATE, -age)
        return calendar
    }

    fun yearToStringWithEra(year: Int): String {
        val cal: Calendar = GregorianCalendar(if (year < 0) year + 1 else year, 1, 1)
        return getDateStringWithSkeletonPattern(cal.time, if (year < 0) "y GG" else "y")
    }

    fun getYearDifferenceString(context: Context, year: Int, languageCode: String): String {
        val diffInYears = Calendar.getInstance()[Calendar.YEAR] - year
        val targetResource = context.getResources(languageCode)
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val firstMatchLocaleInstance = RelativeDateTimeFormatter.getInstance(targetResource.configuration.locales.getFirstMatch(arrayOf(languageCode)))
            when (diffInYears) {
                0 -> firstMatchLocaleInstance.format(RelativeDateTimeFormatter.Direction.THIS, RelativeDateTimeFormatter.AbsoluteUnit.YEAR)
                1 -> firstMatchLocaleInstance.format(RelativeDateTimeFormatter.Direction.LAST, RelativeDateTimeFormatter.AbsoluteUnit.YEAR)
                -1 -> firstMatchLocaleInstance.format(RelativeDateTimeFormatter.Direction.NEXT, RelativeDateTimeFormatter.AbsoluteUnit.YEAR)
                else -> firstMatchLocaleInstance.format(diffInYears.toDouble(), RelativeDateTimeFormatter.Direction.LAST, RelativeDateTimeFormatter.RelativeUnit.YEARS)
            }
        } else {
            return if (diffInYears == 0) context.getString(languageCode, R.string.this_year)
            else targetResource.getQuantityString(R.plurals.diff_years, diffInYears, diffInYears)
        }
    }

    fun startOfYearInMillis(year: Int, zoneId: ZoneId = ZoneId.systemDefault()): Long {
        val localDate = LocalDate.of(year, 1, 1)
        return localDate.atStartOfDay(zoneId).toInstant().toEpochMilli()
    }

    fun endOfYearInMillis(year: Int, zoneId: ZoneId = ZoneId.systemDefault()): Long {
        val localDate = LocalDate.of(year, 12, 31)
        return localDate.atTime(0, 0, 0).atZone(zoneId).toInstant().toEpochMilli()
    }

    fun epochMilliToYear(epochMilli: Long, zoneId: ZoneId = ZoneId.systemDefault()): Int {
        val zonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(epochMilli), zoneId)
        return zonedDateTime.year
    }

    fun toRelativeDateString(date: Date): String {
        val localDate = date.toLocalDate()
        val now = LocalDate.now()
        val daysDiff = ChronoUnit.DAYS.between(localDate, now)

        return when {
            daysDiff < 7 -> {
                val dayOfWeek = localDate.dayOfWeek.getDisplayName(TextStyle.FULL, Locale.getDefault())
                dayOfWeek
            }

            localDate.year == now.year -> {
                DateUtil.getMonthOnlyDateString(localDate)
            }

            else -> {
                DateUtil.getShortDateString(localDate)
            }
        }
    }
}
