package org.wikipedia.compose.components

import androidx.compose.foundation.layout.RowScope
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import org.wikipedia.R
import org.wikipedia.analytics.eventplatform.BreadCrumbLogEvent
import org.wikipedia.compose.theme.WikipediaTheme

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SearchTopAppBar(
    modifier: Modifier = Modifier,
    searchQuery: String,
    placeHolderTitle: String,
    onSearchQueryChange: (String) -> Unit,
    onBackButtonClick: () -> Unit,
    actions: @Composable RowScope.() -> Unit = {},
) {
    val focusRequester = remember { FocusRequester() }
    val context = LocalContext.current

    TopAppBar(
        modifier = modifier
            .fillMaxWidth()
            .focusRequester(focusRequester),
        title = {
            OutlinedTextField(
                modifier = Modifier
                    .padding(top = 2.dp)
                    .testTag("search_text_field"),
                value = searchQuery,
                onValueChange = onSearchQueryChange,
                placeholder = {
                    Text(placeHolderTitle)
                },
                colors = OutlinedTextFieldDefaults.colors(
                    unfocusedBorderColor = Color.Transparent,
                    focusedBorderColor = Color.Transparent,
                    focusedTextColor = WikipediaTheme.colors.primaryColor,
                    cursorColor = WikipediaTheme.colors.progressiveColor
                ),
                textStyle = TextStyle(
                    fontSize = 16.sp
                ),
                singleLine = true
            )
        },
        navigationIcon = {
            IconButton(onClick = {
                BreadCrumbLogEvent.logClick(context, "searchBackButton")
                onBackButtonClick()
            }) {
                Icon(
                    painter = painterResource(R.drawable.ic_arrow_back_black_24dp),
                    contentDescription = stringResource(R.string.search_back_button_content_description),
                    tint = WikipediaTheme.colors.primaryColor
                )
            }
        },
        actions = actions,
        colors = TopAppBarDefaults.topAppBarColors(
            containerColor = WikipediaTheme.colors.paperColor,
            titleContentColor = WikipediaTheme.colors.primaryColor
        ),
    )

    LaunchedEffect(Unit) {
        BreadCrumbLogEvent.logClick(context, "search")
        focusRequester.requestFocus()
    }
}
