package org.wikipedia.donate

import android.content.Context
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.launch
import org.wikipedia.BuildConfig
import org.wikipedia.R
import org.wikipedia.WikipediaApp
import org.wikipedia.activity.BaseActivity
import org.wikipedia.analytics.eventplatform.DonorExperienceEvent
import org.wikipedia.databinding.DialogDonateBinding
import org.wikipedia.dataclient.donate.CampaignCollection
import org.wikipedia.donate.donationreminder.DonationReminderAbTest
import org.wikipedia.donate.donationreminder.DonationReminderHelper
import org.wikipedia.page.ExtendedBottomSheetDialogFragment
import org.wikipedia.settings.Prefs
import org.wikipedia.util.CustomTabsUtil
import org.wikipedia.util.FeedbackUtil
import org.wikipedia.util.Resource

class DonateDialog : ExtendedBottomSheetDialogFragment() {
    private var _binding: DialogDonateBinding? = null
    private val binding get() = _binding!!
    private var campaignIdOriginal: String? = null
    private var campaignId: String? = null

    private val viewModel: DonateViewModel by viewModels()

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        _binding = DialogDonateBinding.inflate(inflater, container, false)
        campaignIdOriginal = arguments?.getString(ARG_CAMPAIGN_ID)
        campaignId = campaignIdOriginal + if (DonationReminderHelper.isInEligibleCountry) {
            if (DonationReminderAbTest().isTestGroupUser()) "_reminderB" else "_reminderA"
        } else ""

        val activeInterface = if (arguments?.getBoolean(ARG_FROM_YIR) == true) {
            "wiki_yir"
        } else {
            if (campaignId.isNullOrEmpty()) "setting" else "article_banner"
        }

        binding.donateOtherButton.setOnClickListener {
            DonorExperienceEvent.logAction("webpay_click", activeInterface, campaignId = campaignId)
            onDonateClicked()
        }

        binding.donateGooglePayButton.setOnClickListener {
            invalidateCampaign()
            DonorExperienceEvent.logAction("gpay_click", activeInterface, campaignId = campaignId)
            (requireActivity() as? BaseActivity)?.launchDonateActivity(
                GooglePayComponent.getDonateActivityIntent(requireActivity(), campaignId, arguments?.getString(ARG_DONATE_URL)))
        }

        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.CREATED) {
                launch {
                    viewModel.uiState.collect {
                        when (it) {
                            is Resource.Loading -> {
                                binding.progressBar.isVisible = true
                                binding.contentsContainer.isVisible = false
                            }

                            is Resource.Error -> {
                                binding.progressBar.isVisible = false
                                FeedbackUtil.showMessage(
                                    this@DonateDialog,
                                    it.throwable.localizedMessage.orEmpty()
                                )
                            }

                            is Resource.Success -> {
                                // if Google Pay is not available, then bounce right out to external workflow.
                                if (!it.data) {
                                    onDonateClicked()
                                    return@collect
                                }
                                binding.progressBar.isVisible = false
                                binding.contentsContainer.isVisible = true
                            }
                        }
                    }
                }
                if (arguments?.getBoolean(ARG_FROM_DONATION_REMINDER) == true) {
                    setupDirectGooglePayButton()
                }
            }
        }

        viewModel.checkGooglePayAvailable(requireActivity())

        return binding.root
    }

    override fun onDestroyView() {
        _binding = null
        super.onDestroyView()
    }

    private fun onDonateClicked() {
        launchDonateLink(requireContext(), url = arguments?.getString(ARG_DONATE_URL), campaignId = campaignId)
        invalidateCampaign()
        dismiss()
    }

    private fun invalidateCampaign() {
        campaignIdOriginal?.let {
            Prefs.announcementShownDialogs = setOf(it)
        }
    }

    private fun setupDirectGooglePayButton() {
        val donateAmount = Prefs.donationReminderConfig.donateAmount
        val donateAmountText =
            DonateUtil.currencyFormat.format(Prefs.donationReminderConfig.donateAmount)
        val donateButtonText = getString(R.string.donation_reminders_gpay_text, donateAmountText)
        binding.donateGooglePayButton.text = donateButtonText
        binding.donateGooglePayButton.setOnClickListener {
            DonorExperienceEvent.logDonationReminderAction(
                activeInterface = "reminder_milestone",
                action = "gpay_click",
                campaignId = DonationReminderHelper.campaignId
            )
            (requireActivity() as? BaseActivity)?.launchDonateActivity(
                GooglePayComponent.getDonateActivityIntent(requireActivity(), filledAmount = donateAmount, campaignId = DonationReminderHelper.campaignId))
        }
        binding.donateGooglePayDifferentAmountButton.isVisible = true
        binding.donateGooglePayDifferentAmountButton.setOnClickListener {
            DonorExperienceEvent.logDonationReminderAction(
                activeInterface = "reminder_milestone",
                action = "other_gpay_click",
                campaignId = DonationReminderHelper.campaignId
            )
            (requireActivity() as? BaseActivity)?.launchDonateActivity(
                GooglePayComponent.getDonateActivityIntent(requireActivity(), campaignId = DonationReminderHelper.campaignId))
        }
        binding.donateOtherButton.setOnClickListener {
            DonorExperienceEvent.logDonationReminderAction(
                activeInterface = "reminder_milestone",
                action = "other_method_click",
                campaignId = DonationReminderHelper.campaignId
            )
            onDonateClicked()
        }
        binding.gPayHeaderContainer.isVisible = false
    }

    companion object {
        const val ARG_CAMPAIGN_ID = "campaignId"
        const val ARG_DONATE_URL = "donateUrl"
        const val ARG_FROM_DONATION_REMINDER = "fromDonationReminder"
        const val ARG_FROM_YIR = "fromYiR"

        fun newInstance(campaignId: String? = null, donateUrl: String? = null, fromDonationReminder: Boolean = false, fromYiR: Boolean = false): DonateDialog {
            return DonateDialog().apply {
                arguments = bundleOf(
                    ARG_CAMPAIGN_ID to campaignId,
                    ARG_DONATE_URL to donateUrl,
                    ARG_FROM_DONATION_REMINDER to fromDonationReminder,
                    ARG_FROM_YIR to fromYiR,
                )
            }
        }

        fun launchDonateLink(context: Context, url: String? = null, campaignId: String? = "appmenu") {
            val formattedCampaignId = campaignId?.let {
                return@let CampaignCollection.getFormattedCampaignId(it)
            }.orEmpty()
            val donateUrl = url ?: context.getString(R.string.donate_url, formattedCampaignId,
                WikipediaApp.instance.languageState.systemLanguageCode, BuildConfig.VERSION_NAME, Prefs.appInstallId)
            CustomTabsUtil.openInCustomTab(context, donateUrl)
        }
    }
}
