package org.wikipedia.settings

import android.content.DialogInterface
import android.content.Intent
import android.os.Build
import androidx.preference.Preference
import androidx.preference.PreferenceFragmentCompat
import androidx.preference.SwitchPreferenceCompat
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import org.wikipedia.BuildConfig
import org.wikipedia.Constants
import org.wikipedia.R
import org.wikipedia.WikipediaApp
import org.wikipedia.analytics.eventplatform.DonorExperienceEvent
import org.wikipedia.analytics.eventplatform.RecommendedReadingListEvent
import org.wikipedia.analytics.eventplatform.YearInReviewEvent
import org.wikipedia.auth.AccountUtil
import org.wikipedia.donate.DonateUtil
import org.wikipedia.donate.donationreminder.DonationReminderActivity
import org.wikipedia.donate.donationreminder.DonationReminderHelper
import org.wikipedia.feed.configure.ConfigureActivity
import org.wikipedia.login.LoginActivity
import org.wikipedia.page.ExclusiveBottomSheetPresenter
import org.wikipedia.readinglist.recommended.RecommendedReadingListOnboardingActivity
import org.wikipedia.readinglist.recommended.RecommendedReadingListSettingsActivity
import org.wikipedia.readinglist.recommended.RecommendedReadingListSource
import org.wikipedia.readinglist.sync.ReadingListSyncAdapter
import org.wikipedia.settings.languages.WikipediaLanguagesActivity
import org.wikipedia.theme.ThemeFittingRoomActivity
import org.wikipedia.util.FeedbackUtil
import org.wikipedia.yearinreview.YearInReviewViewModel

internal class SettingsPreferenceLoader(fragment: PreferenceFragmentCompat) : BasePreferenceLoader(fragment) {
    override fun loadPreferences() {
        loadPreferences(R.xml.preferences)
        if (RemoteConfig.config.disableReadingListSync) {
            findPreference(R.string.preference_category_sync).isVisible = false
            findPreference(R.string.preference_key_sync_reading_lists).isVisible = false
        }
        findPreference(R.string.preference_key_sync_reading_lists).onPreferenceChangeListener = SyncReadingListsListener()
        loadPreferences(R.xml.preferences_about)
        updateLanguagePrefSummary()
        findPreference(R.string.preference_key_language).onPreferenceClickListener = Preference.OnPreferenceClickListener {
            activity.startActivityForResult(WikipediaLanguagesActivity.newIntent(activity, Constants.InvokeSource.SETTINGS),
                    Constants.ACTIVITY_REQUEST_ADD_A_LANGUAGE)
            true
        }
        findPreference(R.string.preference_key_customize_explore_feed).onPreferenceClickListener = Preference.OnPreferenceClickListener {
             activity.startActivityForResult(
                 ConfigureActivity.newIntent(activity, Constants.InvokeSource.NAV_MENU.ordinal),
                    Constants.ACTIVITY_REQUEST_FEED_CONFIGURE)
            true
        }
        findPreference(R.string.preference_key_color_theme).let {
            it.setSummary(WikipediaApp.instance.currentTheme.nameId)
            it.onPreferenceClickListener = Preference.OnPreferenceClickListener {
                activity.startActivity(ThemeFittingRoomActivity.newIntent(activity))
                true
            }
        }

        findPreference(R.string.preference_key_selected_app_icon).let {
            it.isVisible = YearInReviewViewModel.isCustomIconAllowed
            it.summary = fragment.getString(R.string.settings_app_icon_preference_subtitle, YearInReviewViewModel.YIR_YEAR)
            it.onPreferenceClickListener = Preference.OnPreferenceClickListener {
                ExclusiveBottomSheetPresenter.show(fragment.parentFragmentManager, AppIconDialog())
                true
            }
        }

        findPreference(R.string.preference_key_year_in_review_is_enabled).let {
            it.isVisible = YearInReviewViewModel.isAccessible
            if (it.isVisible) {
                YearInReviewEvent.submit(action = "impression", slide = "setting")
            }
            it.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { preference, newValue ->
                if (newValue as Boolean) {
                    YearInReviewEvent.submit(action = "yir_on_click", slide = "setting")
                    return@OnPreferenceChangeListener true
                }
                YearInReviewEvent.submit(action = "yir_off_click", slide = "setting")
                MaterialAlertDialogBuilder(activity)
                    .setTitle(R.string.year_in_review_disable_title)
                    .setMessage(R.string.year_in_review_setting_subtitle)
                    .setPositiveButton(R.string.year_in_review_disable_positive_button) { _, _ ->
                        YearInReviewEvent.submit(action = "yir_off_confirm_click", slide = "setting")
                        Prefs.yearInReviewModelData = emptyMap()
                        Prefs.yearInReviewReadingListSurveyShown = false
                        Prefs.yearInReviewReadingListVisitCount = 0
                        (preference as SwitchPreferenceCompat).isChecked = false
                    }
                    .setNegativeButton(R.string.year_in_review_disable_negative_button) { _, _ ->
                        YearInReviewEvent.submit(action = "yir_off_cancel_click", slide = "setting")
                    }
                    .show()
                false
            }
        }

        findPreference(R.string.preference_key_about_wikipedia_app).onPreferenceClickListener = Preference.OnPreferenceClickListener {
            activity.startActivity(Intent(activity, AboutActivity::class.java))
            true
        }
        findPreference(R.string.preference_key_send_feedback).onPreferenceClickListener =
            Preference.OnPreferenceClickListener {
                FeedbackUtil.composeEmail(
                    activity,
                    subject = "Android App ${BuildConfig.VERSION_NAME} Feedback",
                    body = deviceInformation()
                )
                true
        }
        findPreference(R.string.preference_key_recommended_reading_list_enabled).onPreferenceClickListener = Preference.OnPreferenceClickListener {
            RecommendedReadingListEvent.submit("discover_click", "global_settings")
            if (Prefs.recommendedReadingListInterests.isEmpty() &&
                Prefs.recommendedReadingListSource == RecommendedReadingListSource.INTERESTS) {
                activity.startActivity(RecommendedReadingListOnboardingActivity.newIntent(activity))
            } else {
                activity.startActivity(RecommendedReadingListSettingsActivity.newIntent(activity))
            }
            true
        }

        if (AccountUtil.isLoggedIn) {
            loadPreferences(R.xml.preferences_account)
            (findPreference(R.string.preference_key_logout) as LogoutPreference).activity = activity
        }

        findPreference(R.string.preference_key_donation_reminders).apply {
            isVisible = DonationReminderHelper.isEnabled
            onPreferenceClickListener =
                Preference.OnPreferenceClickListener {
                    activity.startActivity(DonationReminderActivity.newIntent(activity, isFromSettings = true))
                    true
                }
        }
        findPreference(R.string.preference_key_delete_local_donation_history).onPreferenceClickListener = Preference.OnPreferenceClickListener {
            DonorExperienceEvent.logDonationReminderAction(
                activeInterface = "global_setting",
                action = "clear_donation_hist_click"
            )
            val hasDonations = Prefs.donationResults.isNotEmpty()

            if (hasDonations) {
                Prefs.donationResults = emptyList()
            }

            val messageResId = if (hasDonations) {
                R.string.donation_history_deleted_message_snackbar
            } else R.string.donation_history_no_history_message_snackbar
            FeedbackUtil.showMessage(activity, activity.resources.getString(messageResId))
            true
        }
    }

    private fun deviceInformation(): String {
        return "\n\nVersion: ${BuildConfig.VERSION_NAME} \nDevice: ${Build.BRAND} ${Build.MODEL} (SDK: ${Build.VERSION.SDK_INT})\n"
    }

    fun updateLanguagePrefSummary() {
        // TODO: resolve RTL vs LTR with multiple languages (e.g. list contains English and Hebrew)
        findPreference(R.string.preference_key_language).summary = WikipediaApp.instance.languageState.appLanguageLocalizedNames
    }

    fun updateRecommendedReadingListSummary() {
        val summary = if (Prefs.isRecommendedReadingListEnabled) {
            R.string.recommended_reading_list_settings_toggle_enable_message
        } else R.string.recommended_reading_list_settings_toggle_disable_message
        findPreference(R.string.preference_key_recommended_reading_list_enabled).summary = activity.getString(summary)
    }

    fun updateDonationRemindersDescription() {
        val articleFrequency = activity.resources.getQuantityString(R.plurals.donation_reminders_text_articles, Prefs.donationReminderConfig.articleFrequency, Prefs.donationReminderConfig.articleFrequency)
        val description = if (Prefs.donationReminderConfig.userEnabled) activity.getString(R.string.donation_reminders_settings_description_on,
            DonateUtil.currencyFormat.format(Prefs.donationReminderConfig.donateAmount), articleFrequency) else
                activity.getString(R.string.donation_reminders_settings_description_off)
        findPreference(R.string.preference_key_donation_reminders).summary = description
    }

    private inner class SyncReadingListsListener : Preference.OnPreferenceChangeListener {
        override fun onPreferenceChange(preference: Preference, newValue: Any): Boolean {
            if (AccountUtil.isLoggedIn) {
                if (newValue as Boolean) {
                    (preference as SwitchPreferenceCompat).isChecked = true
                    ReadingListSyncAdapter.setSyncEnabledWithSetup()
                } else {
                    MaterialAlertDialogBuilder(activity)
                            .setTitle(activity.getString(R.string.preference_dialog_of_turning_off_reading_list_sync_title, AccountUtil.userName))
                            .setMessage(activity.getString(R.string.preference_dialog_of_turning_off_reading_list_sync_text, AccountUtil.userName))
                            .setPositiveButton(R.string.reading_lists_confirm_remote_delete_yes, DeleteRemoteListsYesListener(preference))
                            .setNegativeButton(R.string.reading_lists_confirm_remote_delete_no, null)
                            .show()
                }
            } else {
                MaterialAlertDialogBuilder(activity)
                        .setTitle(R.string.reading_list_preference_login_to_enable_sync_dialog_title)
                        .setMessage(R.string.reading_list_preference_login_to_enable_sync_dialog_text)
                        .setPositiveButton(R.string.reading_list_preference_login_to_enable_sync_dialog_login
                        ) { _: DialogInterface, _: Int ->
                            val loginIntent = LoginActivity.newIntent(activity,
                                    LoginActivity.SOURCE_SETTINGS)
                            activity.startActivity(loginIntent)
                        }
                        .setNegativeButton(R.string.reading_list_preference_login_to_enable_sync_dialog_cancel, null)
                        .show()
            }
            // clicks are handled and preferences updated accordingly; don't pass the result through
            return false
        }
    }

    fun updateSyncReadingListsPrefSummary() {
        findPreference(R.string.preference_key_sync_reading_lists).let {
            if (AccountUtil.isLoggedIn) {
                it.summary = activity.getString(R.string.preference_summary_sync_reading_lists_from_account, AccountUtil.userName)
            } else {
                it.setSummary(R.string.preference_summary_sync_reading_lists)
            }
        }
    }

    private class DeleteRemoteListsYesListener(private val preference: Preference) : DialogInterface.OnClickListener {
        override fun onClick(dialog: DialogInterface, which: Int) {
            (preference as SwitchPreferenceCompat).isChecked = false
            Prefs.isReadingListSyncEnabled = false
            Prefs.isReadingListsRemoteSetupPending = false
            Prefs.isReadingListsRemoteDeletePending = true
            ReadingListSyncAdapter.manualSync()
        }
    }
}
