package org.wikipedia.robots.feature

import BaseRobot
import android.content.Context
import android.util.Log
import androidx.annotation.IdRes
import androidx.compose.ui.test.onNodeWithContentDescription
import androidx.compose.ui.test.performClick
import androidx.recyclerview.widget.RecyclerView
import androidx.test.espresso.Espresso.onView
import androidx.test.espresso.ViewAction
import androidx.test.espresso.action.ViewActions.click
import androidx.test.espresso.action.ViewActions.scrollTo
import androidx.test.espresso.assertion.ViewAssertions.doesNotExist
import androidx.test.espresso.assertion.ViewAssertions.matches
import androidx.test.espresso.contrib.RecyclerViewActions
import androidx.test.espresso.contrib.RecyclerViewActions.actionOnItemAtPosition
import androidx.test.espresso.matcher.ViewMatchers.hasDescendant
import androidx.test.espresso.matcher.ViewMatchers.isDisplayed
import androidx.test.espresso.matcher.ViewMatchers.withContentDescription
import androidx.test.espresso.matcher.ViewMatchers.withId
import androidx.test.espresso.matcher.ViewMatchers.withParent
import androidx.test.espresso.matcher.ViewMatchers.withText
import org.hamcrest.Matchers.allOf
import org.junit.Assert.assertTrue
import org.wikipedia.R
import org.wikipedia.TestUtil.childAtPosition
import org.wikipedia.base.TestConfig

class SettingsRobot : BaseRobot() {
    fun verifyTitle() = apply {
        verify.viewWithTextDisplayed("Settings")
    }

    fun clickExploreFeedSettingItem() = apply {
        // Click on `Explore feed` option
        onView(
            allOf(
                withId(R.id.recycler_view),
            childAtPosition(withId(android.R.id.list_container), 0)
            )
        )
            .perform(actionOnItemAtPosition<RecyclerView.ViewHolder>(2, click()))

        delay(TestConfig.DELAY_MEDIUM)
    }

    fun openMoreOptionsToolbar() = apply {
        onView(allOf(
            withContentDescription("More options"),
            childAtPosition(childAtPosition(withId(R.id.toolbar), 2), 0), isDisplayed()
        ))
            .perform(click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun hideAllExploreFeeds() = apply {
        // Choose the option to hide all explore feed cards
        onView(allOf(withId(R.id.title), withText("Hide all"),
            childAtPosition(childAtPosition(withId(androidx.appcompat.R.id.content), 0), 0), isDisplayed()))
            .perform(click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun showAllExploreFeeds() = apply {
        onView(allOf(withId(R.id.title), withText("Show all"),
            childAtPosition(childAtPosition(withId(androidx.appcompat.R.id.content), 0), 0), isDisplayed()))
            .perform(click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickAboutWikipediaAppOptionItem() = apply {
        scrollToSettingsPreferenceItem(R.string.about_description, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun activateDeveloperMode(context: Context) = apply {
        // Click 7 times to activate developer mode
        for (i in 1 until 8) {
            composeTestRule.onNodeWithContentDescription(context.getString(R.string.about_logo_content_description))
                .performClick()
            delay(TestConfig.DELAY_MEDIUM)
        }
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickDeveloperMode() = apply {
        // Assert that developer mode is activated
        onView(allOf(withId(R.id.developer_settings), withContentDescription("Developer settings"),
            childAtPosition(childAtPosition(withId(androidx.appcompat.R.id.action_bar), 2), 0), isDisplayed()))
            .perform(click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun assertWeAreInDeveloperSettings() = apply {
        onView(allOf(withText("Developer settings"),
            withParent(allOf(withId(androidx.appcompat.R.id.action_bar),
                withParent(withId(androidx.appcompat.R.id.action_bar_container))
            )), isDisplayed()))
            .check(matches(withText("Developer settings")))
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickLanguages() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_language, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickExploreFeed() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_customize_explore_feed, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickLogOut(context: Context) = apply {
        try {
            scrollToSettingsPreferenceItem(R.string.preference_title_logout, scrollTo())
            click.onViewWithText(context.getString(R.string.preference_title_logout))
            delay(TestConfig.DELAY_MEDIUM)
        } catch (e: Exception) {
            Log.e("SettingsRobotError:", "User is not logged in.")
        }
    }

    fun toggleShowLinkPreviews() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_show_link_previews, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun toggleCollapseTables() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_collapse_tables, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun clickAppTheme() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_app_theme, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun toggleDownloadReadingList() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_download_reading_list_articles, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun toggleShowImages() = apply {
        scrollToSettingsPreferenceItem(R.string.preference_title_show_images, click())
        delay(TestConfig.DELAY_MEDIUM)
    }

    fun verifyExploreFeedIsEmpty(context: Context) = apply {
        try {
            verify.viewWithTextDisplayed(text = context.getString(R.string.feed_empty_message))
            delay(TestConfig.DELAY_SHORT)
        } catch (e: AssertionError) {
            Log.d("SettingsRobot: ", "Assertion error due to offline mode")
            // checks offline card is visible
           verify.viewWithTextDisplayed(context.getString(R.string.view_offline_card_text))
            // test the feed is empty
            onView(withId(R.id.feed_view))
                .check { view, noViewFoundException ->
                    val expectedCount = 2
                    val recyclerView = view as RecyclerView
                    val itemCount = recyclerView.adapter?.itemCount ?: 0
                    assertTrue("ExpectedCount: $expectedCount, Actual: $itemCount", itemCount == expectedCount)
                }
            onView(withText("Featured article")).check(doesNotExist())
        }
    }

    fun verifyExploreFeedIsNotEmpty(context: Context) = apply {
        verify.textIsNotVisible(context.getString(R.string.feed_empty_message))
        delay(TestConfig.DELAY_SHORT)
    }

    fun pressBack() = apply {
        goBack()
        delay(TestConfig.DELAY_MEDIUM)
    }

    private fun scrollToSettingsPreferenceItem(@IdRes preferenceTitle: Int, viewAction: ViewAction) = apply {
        onView(withId(androidx.preference.R.id.recycler_view))
            .perform(
                RecyclerViewActions.actionOnItem<RecyclerView.ViewHolder>
                    (hasDescendant(withText(preferenceTitle)), viewAction))
        delay(TestConfig.DELAY_MEDIUM)
    }
}
