package org.wikipedia.database

import android.content.Context
import androidx.room.Room
import androidx.test.core.app.ApplicationProvider
import androidx.test.ext.junit.runners.AndroidJUnit4
import androidx.test.platform.app.InstrumentationRegistry
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import org.hamcrest.CoreMatchers.equalTo
import org.hamcrest.CoreMatchers.notNullValue
import org.hamcrest.CoreMatchers.nullValue
import org.hamcrest.MatcherAssert.assertThat
import org.junit.After
import org.junit.Before
import org.junit.Test
import org.junit.runner.RunWith
import org.wikipedia.json.JsonUtil
import org.wikipedia.notifications.db.Notification
import org.wikipedia.notifications.db.NotificationDao
import org.wikipedia.search.db.RecentSearch
import org.wikipedia.search.db.RecentSearchDao
import org.wikipedia.talk.db.TalkPageSeen
import org.wikipedia.talk.db.TalkPageSeenDao
import org.wikipedia.util.log.L
import java.util.Date

@RunWith(AndroidJUnit4::class)
class AppDatabaseTests {
    private lateinit var db: AppDatabase
    private lateinit var recentSearchDao: RecentSearchDao
    private lateinit var talkPageSeenDao: TalkPageSeenDao
    private lateinit var notificationDao: NotificationDao

    @Before
    fun createDb() {
        val context = ApplicationProvider.getApplicationContext<Context>()
        db = Room.inMemoryDatabaseBuilder(context, AppDatabase::class.java).build()
        recentSearchDao = db.recentSearchDao()
        talkPageSeenDao = db.talkPageSeenDao()
        notificationDao = db.notificationDao()
    }

    @After
    fun closeDb() {
        db.close()
    }

    @Test
    fun testRecentSearch() = runBlocking {
        val now = Date()

        recentSearchDao.insertRecentSearch(RecentSearch("Foo"))
        recentSearchDao.insertRecentSearch(RecentSearch("Bar", now))

        var results = recentSearchDao.getRecentSearches()
        assertThat(results.size, equalTo(2))
        assertThat(results[0].text, equalTo("Foo"))
        assertThat(results[1].text, equalTo("Bar"))
        assertThat(results[1].timestamp, equalTo(now))

        recentSearchDao.insertRecentSearch(RecentSearch("Baz"))
        results = recentSearchDao.getRecentSearches()
        assertThat(results.size, equalTo(3))

        recentSearchDao.deleteAll()
        results = recentSearchDao.getRecentSearches()
        assertThat(results.size, equalTo(0))
    }

    @Test
    fun testTalkPageSeen() {
        CoroutineScope(Dispatchers.Default).launch(CoroutineExceptionHandler { _, msg ->
            run {
                L.e(msg)
            }
        }) { withContext(Dispatchers.Main) {
            talkPageSeenDao.insertTalkPageSeen(TalkPageSeen("328b389f2063da236be9d363b272eb0fa6e065816607099c7db8c09e1c919617"))
            talkPageSeenDao.insertTalkPageSeen(TalkPageSeen("5fbbb2d46ead3355750e90032feb34051a552a6f1c76cf1b4072d8d158af9de7"))
            assertThat(talkPageSeenDao.getTalkPageSeen("328b389f2063da236be9d363b272eb0fa6e065816607099c7db8c09e1c919617"), notNullValue())
            assertThat(talkPageSeenDao.getTalkPageSeen("foo"), nullValue())

            var allSeen = talkPageSeenDao.getAll()
            assertThat(allSeen.count(), equalTo(2))

            talkPageSeenDao.deleteAll()
            allSeen = talkPageSeenDao.getAll()
            assertThat(allSeen.count(), equalTo(0))
        }
        }
    }

    @Test
    fun testNotification() = runBlocking {
        val rawJson = InstrumentationRegistry.getInstrumentation()
            .context.resources.assets.open("database/json/notifications.json")
            .bufferedReader()
            .use { it.readText() }

        val notifications = JsonUtil.decodeFromString<List<Notification>>(rawJson)!!

        notificationDao.insertNotifications(notifications)

        var enWikiList = notificationDao.getNotificationsByWiki(listOf("enwiki"))
        val zhWikiList = notificationDao.getNotificationsByWiki(listOf("zhwiki"))
        assertThat(enWikiList, notNullValue())
        assertThat(enWikiList.first().id, equalTo(123759827))
        assertThat(zhWikiList.first().id, equalTo(2470933))
        assertThat(enWikiList.first().isUnread, equalTo(false))
        assertThat(enWikiList.size, equalTo(2))
        assertThat(notificationDao.getAllNotifications().size, equalTo(3))

        val firstEnNotification = enWikiList.first()
        firstEnNotification.read = null
        notificationDao.updateNotification(firstEnNotification)

        // get updated item
        enWikiList = notificationDao.getNotificationsByWiki(listOf("enwiki"))
        assertThat(enWikiList.first().id, equalTo(123759827))
        assertThat(enWikiList.first().isUnread, equalTo(true))

        notificationDao.deleteNotification(firstEnNotification)
        assertThat(notificationDao.getAllNotifications().size, equalTo(2))
        assertThat(notificationDao.getNotificationsByWiki(listOf("enwiki")).size, equalTo(1))

        notificationDao.deleteNotification(notificationDao.getNotificationsByWiki(listOf("enwiki")).first())
        assertThat(notificationDao.getNotificationsByWiki(listOf("enwiki")).isEmpty(), equalTo(true))

        notificationDao.deleteNotification(notificationDao.getNotificationsByWiki(listOf("zhwiki")).first())
        assertThat(notificationDao.getAllNotifications().isEmpty(), equalTo(true))
    }
}
