@file:Suppress("ktlint:standard:no-wildcard-imports")

package org.unifiedpush.distributor.nextpush.services

import android.content.Context
import android.util.Log
import androidx.work.*
import java.util.Calendar
import org.unifiedpush.distributor.WorkerCompanion
import org.unifiedpush.distributor.nextpush.AppCompanion
import org.unifiedpush.distributor.nextpush.Distributor
import org.unifiedpush.distributor.nextpush.account.AccountFactory.getAccount
import org.unifiedpush.distributor.nextpush.callback.NetworkCallbackFactory
import org.unifiedpush.distributor.nextpush.utils.TAG

class RestartWorker(ctx: Context, params: WorkerParameters) : Worker(ctx, params) {

    /**
     * Restart the service if we have never received an event, or haven't received an event
     * in the expected time
     */
    override fun doWork(): Result {
        // We avoid running twice at the same time
        synchronized(lock) {
            Log.d(TAG, "Working")
            if (!NetworkCallbackFactory.hasInternet()) {
                Log.d(TAG, "Aborting, no internet.")
                return Result.success()
            }
            val currentDate = Calendar.getInstance()
            val restartDate = Calendar.getInstance()
            AppCompanion.lastEventDate?.let {
                // We check that the last received event isn't older than the keep alive
                restartDate.time = it.time
                restartDate.add(Calendar.MILLISECOND, AppCompanion.keepalive.get())
                Log.d(TAG, "restartDate: ${restartDate.time}")

                /**
                 * We restart the service when:
                 * - we didn't receive the last ping
                 * - we have received more than [MAX_MSG] messages, so the received messages can be ack to the server
                 * and deleted from the server
                 */
                val timeoutExpire = currentDate.after(restartDate)
                val msgCount = AppCompanion.messageCounter.get()
                if (timeoutExpire || msgCount > MAX_MSG) {
                    if (timeoutExpire) {
                        Log.d(TAG, "We should have received an event before ${restartDate.time}. Restarting")
                    } else {
                        Log.d(TAG, "Restarting to acknowledge $msgCount messages.")
                    }
                    /**
                     * If there is at least one failure, we do not update [SourceManager] failures' counter,
                     * it will be done by the next requests.
                     * Else we add one failure to be sure the service restarts.
                     */
                    if (SourceManager.isRunningWithoutFailure) {
                        SourceManager.setFailOnce()
                    }
                    StartService.startService(applicationContext)
                    // We consider this run as the first event
                    AppCompanion.lastEventDate = currentDate
                }
            } ?: run {
                Log.d(TAG, "Restarting")
                StartService.startService(applicationContext)
                // We consider this run as the first event
                AppCompanion.lastEventDate = currentDate
            }
        }
        return Result.success()
    }

    companion object : WorkerCompanion(RestartWorker::class.java) {
        private val lock = Object()

        /**
         * Number of messages before restarting the service to acknowledge to the server received messages.
         *
         * 256 messages means a _maximum_ of 1MB of ack messages (if all of them are 4kB)
         */
        private const val MAX_MSG = 256
        override fun canRun(context: Context): Boolean = getAccount(context) != null

        override fun isServiceStarted(context: Context): Boolean = StartService.isServiceStarted()

        override fun enableComponents(context: Context) {
            Distributor.enableComponents(context)
        }

        override fun disableComponents(context: Context) {
            Distributor.disableComponents(context)
        }

        override fun run(context: Context, delay: Long) {
            AppCompanion.lastEventDate = null
            super.run(context, delay)
        }
    }
}
