package org.unifiedpush.distributor.nextpush.api

import android.content.Context
import android.util.Base64
import android.util.Log
import com.google.gson.Gson
import java.lang.Exception
import java.util.Calendar
import java.util.Timer
import java.util.TimerTask
import java.util.concurrent.atomic.AtomicReference
import kotlin.concurrent.schedule
import okhttp3.Response
import okhttp3.sse.EventSource
import okhttp3.sse.EventSourceListener
import org.unifiedpush.distributor.WakeLock
import org.unifiedpush.distributor.nextpush.AppCompanion
import org.unifiedpush.distributor.nextpush.AppStore
import org.unifiedpush.distributor.nextpush.DatabaseFactory
import org.unifiedpush.distributor.nextpush.Distributor
import org.unifiedpush.distributor.nextpush.Distributor.sendMessage
import org.unifiedpush.distributor.nextpush.LastEventId
import org.unifiedpush.distributor.nextpush.api.response.SSEResponse
import org.unifiedpush.distributor.nextpush.callback.NetworkCallbackFactory
import org.unifiedpush.distributor.nextpush.services.RestartWorker
import org.unifiedpush.distributor.nextpush.services.SourceManager
import org.unifiedpush.distributor.nextpush.services.StartService
import org.unifiedpush.distributor.nextpush.utils.LowKeepAliveNotification
import org.unifiedpush.distributor.nextpush.utils.NoStartNotification
import org.unifiedpush.distributor.nextpush.utils.TAG
import org.unifiedpush.distributor.receiver.DistributorReceiver
import org.unifiedpush.distributor.utils.addOnce
import org.unifiedpush.distributor.utils.removeSync

class SSEListener(private val context: Context, private val releaseLock: () -> Unit) : EventSourceListener() {

    private val store = AppStore(context)

    override var idleTimeoutMillis: Long? = AppCompanion.keepalive.get().toLong() + TIMEOUT_TOLERANCE

    override fun onOpen(eventSource: EventSource, response: Response) {
        StartingTimer.scheduleNewTimer(context, eventSource)
        releaseLock()
        try {
            Log.d(TAG, "onOpen: " + response.code)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    override fun onEvent(
        eventSource: EventSource,
        id: String?,
        type: String?,
        data: String
    ) {
        WakeLock.withWakeLock {
            Log.d(TAG, "New SSE message event: $type")
            AppCompanion.lastEventDate = Calendar.getInstance()

            when (type) {
                "start" -> {
                    StartingTimer.stop()
                    SourceManager.debugStarted()
                    SourceManager.setConnected(context, eventSource)
                    AppCompanion.messageCounter.set(0)
                    AppCompanion.bufferedResponseChecked.set(true)
                    NoStartNotification(context).delete()
                }

                "ping" -> {
                    SourceManager.debugNewPing(context)
                }

                "keepalive" -> {
                    val message = Gson().fromJson(data, SSEResponse::class.java)
                    message.keepalive.let {
                        val keepaliveMs = it * 1000
                        AppCompanion.keepalive.set(keepaliveMs)
                        idleTimeoutMillis = keepaliveMs.toLong() + TIMEOUT_TOLERANCE
                        Log.d(TAG, "New keepalive: $it")
                        if (it < 25) {
                            LowKeepAliveNotification(context, it).showSingle()
                        } else {
                            LowKeepAliveNotification(context, it).delete()
                        }
                    }
                }

                "message" -> {
                    AppCompanion.messageCounter.incrementAndGet()
                    val message = Gson().fromJson(data, SSEResponse::class.java)
                    sendMessage(
                        context,
                        message.token,
                        Base64.decode(message.message, Base64.DEFAULT)
                    )
                    id?.let { LastEventId(store).save(it) }
                }

                "deleteApp" -> {
                    val message = Gson().fromJson(data, SSEResponse::class.java)
                    val connectionToken = DatabaseFactory.getDb(context).getConnectorToken(message.token)
                    if (connectionToken != null && DistributorReceiver.delQueue.addOnce(message.token)) {
                        Distributor.deleteChannelFromServer(context, message.token)
                        DistributorReceiver.delQueue.removeSync(message.token)
                    }
                }
            }
        }
    }

    override fun onClosed(eventSource: EventSource) {
        Log.d(TAG, "onClosed: $eventSource")
        eventSource.cancel()
        releaseLock()
        if (!shouldRestart()) return
        if (SourceManager.addFail(context, eventSource)) {
            clearDebugVars()
            RestartWorker.run(context, delay = 0)
        }
    }

    override fun onFailure(
        eventSource: EventSource,
        t: Throwable?,
        response: Response?
    ) {
        Log.d(TAG, "onFailure")
        eventSource.cancel()
        releaseLock()
        if (!shouldRestart()) return
        t?.let {
            Log.d(TAG, "An error occurred: $t")
        }
        response?.let {
            Log.d(TAG, "onFailure: ${it.code}")
        }
        if (!NetworkCallbackFactory.hasInternet()) {
            Log.d(TAG, "No Internet: do not restart")
            // It will be restarted when Internet is back
            eventSource.cancel()
            clearDebugVars()
            return
        }
        if (SourceManager.addFail(context, eventSource)) {
            clearDebugVars()
            // If there is no delay, keep the periodic worker with
            // its 16 mins
            val delay = SourceManager.getTimeout() ?: return
            Log.d(TAG, "Retrying in $delay ms")
            RestartWorker.run(context, delay = delay)
        }
    }

    /**
     * Check if service is started and call [clearDebugVars] if the service has not started.
     *
     * @return [StartService.isServiceStarted]
     */
    private fun shouldRestart(): Boolean {
        if (!StartService.isServiceStarted()) {
            Log.d(TAG, "StartService not started")
            clearDebugVars()
            return false
        }
        return true
    }

    /**
     * Remove [StartingTimer], set the [SourceManager] debug vars to their default.
     * The startingTimer to check buffering response, and started/ping to check the reverse proxy
     * timeout is high enough.
     */
    private fun clearDebugVars() {
        StartingTimer.stop()
        SourceManager.debugClear()
    }

    /**
     * Timer to stop the service if we don't receive the `start` event within 45 seconds.
     * Used to check if there is a reverse proxy buffering responses.
     */
    private object StartingTimer {
        private var startingTimer: AtomicReference<TimerTask?> = AtomicReference(null)

        fun scheduleNewTimer(context: Context, eventSource: EventSource) {
            val timer: TimerTask? = if (
                !AppCompanion.bufferedResponseChecked.get() &&
                !AppCompanion.booting.getAndSet(false)
            ) {
                Timer().schedule(45_000L) {
                    // 45 secs
                    if (SourceManager.addFail(context, eventSource)) {
                        StartService.stopService()
                        NoStartNotification(context).showSingle()
                    }
                }
            } else {
                null
            }
            startingTimer.getAndSet(timer)?.cancel()
        }

        fun stop() {
            startingTimer.getAndSet(null)?.cancel()
        }
    }

    private companion object {
        private const val TIMEOUT_TOLERANCE = 5_000L // 5 seconds
    }
}
