@file:Suppress("ktlint:standard:no-wildcard-imports")

package org.unifiedpush.distributor.nextpush.api

import android.content.Context
import android.os.Build
import android.util.Log
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.observers.DisposableObserver
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit
import okhttp3.Request
import okhttp3.sse.EventSources
import org.unifiedpush.distributor.nextpush.AppStore
import org.unifiedpush.distributor.nextpush.LastEventId
import org.unifiedpush.distributor.nextpush.account.AccountFactory.getAccount
import org.unifiedpush.distributor.nextpush.account.AccountType
import org.unifiedpush.distributor.nextpush.api.provider.*
import org.unifiedpush.distributor.nextpush.api.provider.ApiProvider.Companion.API_PATH
import org.unifiedpush.distributor.nextpush.api.response.ApiResponse
import org.unifiedpush.distributor.nextpush.callback.BatteryCallbackFactory
import org.unifiedpush.distributor.nextpush.services.SourceManager
import org.unifiedpush.distributor.nextpush.utils.NextPushHttpClient
import org.unifiedpush.distributor.nextpush.utils.NoServerAppNotification

class Api(context: Context) {

    private val context = context.applicationContext
    private val store = AppStore(context)

    private val baseUrl: String
        get() = getAccount(context)?.url ?: "http://0.0.0.0/"

    private fun withApiProvider(block: (ApiProvider, then: () -> Unit) -> Unit) {
        when (store.account.accountType) {
            AccountType.SSO -> ApiSSOFactory(context)
            AccountType.Direct -> ApiDirectFactory(context)
        }.getProviderAndExecute(block)
    }

    private fun tryWithDeviceId(block: (String) -> Unit) {
        store.deviceId?.let {
            block(it)
        }
            ?: run {
                Log.d(TAG, "No deviceId found.")

                val parameters = mapOf("deviceName" to Build.MODEL)
                try {
                    withApiProvider { apiProvider, then ->
                        apiProvider.createDevice(parameters)
                            ?.subscribeOn(Schedulers.io())
                            ?.observeOn(AndroidSchedulers.mainThread())
                            ?.subscribe(object : DisposableObserver<ApiResponse>() {
                                override fun onNext(response: ApiResponse) {
                                    response.deviceId.let {
                                        store.deviceId = it
                                    }
                                }

                                override fun onError(e: Throwable) {
                                    NoServerAppNotification(context).showBig()
                                    e.printStackTrace()
                                    then()
                                }

                                override fun onComplete() {
                                    // Sync once it is registered
                                    store.deviceId?.let {
                                        block(it)
                                    }
                                    Log.d(TAG, "mApi register: onComplete")
                                    then()
                                }
                            })
                    }
                } catch (e: NoProviderException) {
                    e.printStackTrace()
                }
            }
    }

    fun apiDestroy() {
        Log.d(TAG, "Destroying API")
        SourceManager.removeSource()
    }

    fun apiSync(releaseLock: () -> Unit) {
        tryWithDeviceId { deviceId ->
            val client = NextPushHttpClient.Builder()
                .readTimeout(0, TimeUnit.SECONDS)
                .retryOnConnectionFailure(false)
                .build()
            val url = "$baseUrl${API_PATH}device/$deviceId"

            val request = Request.Builder().url(url)
                .get()
                .apply {
                    LastEventId(store).get()?.let {
                        header("Last-Event-ID", it)
                    }
                    if (BatteryCallbackFactory.isLowBattery()) {
                        Log.d(TAG, "Battery is low, registering normal and high only")
                        header("Urgency", "normal")
                    } else {
                        Log.d(TAG, "Battery is OK, registering for all messages")
                    }
                }
                .build()

            EventSources.createFactory(client).newEventSource(
                request,
                SSEListener(context, releaseLock)
            ).also { eventSource ->
                // This will cancel the old eventSource if it exists
                SourceManager.newSource(eventSource)
            }
            Log.d(TAG, "cSync done.")
        }
    }

    fun apiDeleteDevice(block: () -> Unit = {}) {
        tryWithDeviceId { deviceId ->
            try {
                withApiProvider { apiProvider, then ->
                    apiProvider.deleteDevice(deviceId)
                        ?.subscribeOn(Schedulers.io())
                        ?.observeOn(AndroidSchedulers.mainThread())
                        ?.subscribe(object : DisposableObserver<ApiResponse>() {
                            override fun onNext(response: ApiResponse) {
                                if (response.success) {
                                    Log.d(TAG, "Device successfully deleted.")
                                } else {
                                    Log.d(TAG, "An error occurred while deleting the device.")
                                }
                            }

                            override fun onError(e: Throwable) {
                                e.printStackTrace()
                                block()
                                then()
                            }

                            override fun onComplete() {
                                block()
                                then()
                            }
                        })
                    store.deviceId = null
                }
            } catch (e: NoProviderException) {
                e.printStackTrace()
            }
        }
    }

    fun apiCreateApp(
        appName: String,
        vapid: String?,
        onSuccess: (String) -> Unit,
        onError: () -> Unit
    ) {
        tryWithDeviceId { deviceId ->
            // The unity of connector token must already be checked here
            val parameters = mutableMapOf(
                "deviceId" to deviceId,
                "appName" to appName
            )
            vapid?.let {
                parameters.put("vapid", it)
            }
            try {
                withApiProvider { apiProvider, then ->
                    apiProvider.createApp(parameters)
                        ?.subscribeOn(Schedulers.io())
                        ?.observeOn(AndroidSchedulers.mainThread())
                        ?.subscribe(object : DisposableObserver<ApiResponse>() {
                            override fun onNext(response: ApiResponse) {
                                if (response.success) {
                                    Log.d(TAG, "App successfully created.")
                                    onSuccess(response.token)
                                } else {
                                    Log.d(TAG, "An error occurred while creating the application.")
                                    onError()
                                }
                            }

                            override fun onError(e: Throwable) {
                                e.printStackTrace()
                                onError()
                                then()
                            }

                            override fun onComplete() {
                                then()
                            }
                        })
                }
            } catch (e: NoProviderException) {
                e.printStackTrace()
            }
        }
    }

    fun apiDeleteApp(nextpushToken: String, block: () -> Unit) {
        try {
            withApiProvider { apiProvider, then ->
                apiProvider.deleteApp(nextpushToken)
                    ?.subscribeOn(Schedulers.io())
                    ?.observeOn(AndroidSchedulers.mainThread())
                    ?.subscribe(object : DisposableObserver<ApiResponse>() {
                        override fun onNext(response: ApiResponse) {
                            if (response.success) {
                                Log.d(TAG, "App successfully deleted.")
                            } else {
                                Log.d(TAG, "An error occurred while deleting the application.")
                            }
                        }

                        override fun onError(e: Throwable) {
                            e.printStackTrace()
                            then()
                        }

                        override fun onComplete() {
                            block()
                            then()
                        }
                    })
            }
        } catch (e: NoProviderException) {
            e.printStackTrace()
        }
    }

    companion object {
        private const val TAG = "Api"
    }
}
