package org.unifiedpush.distributor.nextpush.activities.ui

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.compose.LocalLifecycleOwner
import androidx.lifecycle.repeatOnLifecycle
import androidx.lifecycle.viewmodel.compose.viewModel
import org.unifiedpush.android.distributor.ui.compose.DynamicColorsPreferences
import org.unifiedpush.android.distributor.ui.compose.Heading
import org.unifiedpush.android.distributor.ui.compose.MigrationPreferences
import org.unifiedpush.android.distributor.ui.compose.MigrationSelectionDialog
import org.unifiedpush.android.distributor.ui.compose.Preference
import org.unifiedpush.android.distributor.ui.compose.ShowToastsPreference
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.activities.DistribMigrationViewModel
import org.unifiedpush.distributor.nextpush.activities.LoginViewModel
import org.unifiedpush.distributor.nextpush.activities.PreviewFactory
import org.unifiedpush.distributor.nextpush.activities.SettingsViewModel
import org.unifiedpush.distributor.nextpush.activities.ThemeViewModel

@Composable
fun SettingsScreen(
    viewModel: SettingsViewModel,
    themeViewModel: ThemeViewModel,
    loginViewModel: LoginViewModel,
    migrationViewModel: DistribMigrationViewModel
) {
    var showLogoutDialog by remember { mutableStateOf(false) }
    var showLogoutMigration by remember { mutableStateOf(false) }
    val lifecycleOwner = LocalLifecycleOwner.current
    LaunchedEffect(Unit) {
        lifecycleOwner.lifecycle.repeatOnLifecycle(Lifecycle.State.RESUMED) {
            migrationViewModel.refreshDistributors()
        }
    }
    Column(
        modifier = Modifier
            .padding(16.dp),
        horizontalAlignment = Alignment.Start,
        verticalArrangement = Arrangement.spacedBy(20.dp)
    ) {
        Heading(R.string.app_name)

        ShowToastsPreference(viewModel.state.showToasts) {
            viewModel.toggleShowToasts()
        }

        DynamicColorsPreferences(themeViewModel.dynamicColors) {
            themeViewModel.toggleDynamicColors()
        }

        Preference(
            stringResource(R.string.logout),
            onclickLabel = stringResource(R.string.open_logout_clicklabel),
            warning = true,
            onSelect = {
                showLogoutDialog = true
            }
        )

        MigrationPreferences(migrationViewModel)
    }

    if (showLogoutDialog) {
        LogoutConfirmationDialog(
            onDismissRequest = {
                showLogoutDialog = false
            },
            onConfirmation = {
                showLogoutDialog = false
                if (migrationViewModel.state.showMigrations) {
                    showLogoutMigration = true
                } else {
                    loginViewModel.logout()
                }
            }
        )
    }
    if (showLogoutMigration) {
        MigrationSelectionDialog(
            migrationViewModel.state.distributors,
            onDismiss = {
                showLogoutMigration = false
                loginViewModel.logout()
            },
            onSelect = { d ->
                showLogoutMigration = false
                migrationViewModel.migrateTo(d)
                loginViewModel.logout()
            }
        )
    }
}

@Preview
@Composable
fun PreviewSettingsScreen() {
    val factory = PreviewFactory(LocalContext.current)
    val settVM = viewModel<SettingsViewModel>(factory = factory)
    val loginVM = viewModel<LoginViewModel>(factory = factory)
    val themeVM = viewModel<ThemeViewModel>(factory = factory)
    val migrationVM = viewModel<DistribMigrationViewModel>(factory = factory)
    SettingsScreen(settVM, themeVM, loginVM, migrationVM)
}
