package org.unifiedpush.distributor.nextpush.activities.ui

import android.annotation.SuppressLint
import androidx.annotation.StringRes
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.ExitTransition
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideInHorizontally
import androidx.compose.animation.slideOutHorizontally
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.ime
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawing
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.tooling.preview.Preview
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import org.unifiedpush.android.distributor.ui.R as LibR
import org.unifiedpush.android.distributor.ui.compose.AppBar
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.activities.DistribMigrationViewModel
import org.unifiedpush.distributor.nextpush.activities.LoginViewModel
import org.unifiedpush.distributor.nextpush.activities.MainViewModel
import org.unifiedpush.distributor.nextpush.activities.PreviewFactory
import org.unifiedpush.distributor.nextpush.activities.SettingsViewModel
import org.unifiedpush.distributor.nextpush.activities.StartViewModel
import org.unifiedpush.distributor.nextpush.activities.ThemeViewModel

enum class AppScreen(@StringRes val title: Int) {
    Login(R.string.app_name),
    Main(R.string.app_name),
    Settings(LibR.string.settings)
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun DefaultTopBar(
    currentScreen: AppScreen,
    canNavigateBack: Boolean,
    navigateUp: () -> Unit,
    modifier: Modifier = Modifier
) = AppBar(
    currentScreen.title,
    canNavigateBack,
    navigateUp,
    modifier
)

private enum class Dir {
    Left,
    Right
}

private fun Dir.transform(it: Int): Int = when (this) {
    Dir.Left -> it
    Dir.Right -> -it
}

private fun slideInTo(dir: Dir): EnterTransition = slideInHorizontally(
    animationSpec = tween(durationMillis = 200)
) { dir.transform(it) } + fadeIn(initialAlpha = 1f)

private fun slideOutFrom(dir: Dir): ExitTransition = slideOutHorizontally(
    animationSpec = tween(durationMillis = 200)
) { dir.transform(it) } + fadeOut(targetAlpha = 1f)

@Composable
fun App(
    factory: ViewModelProvider.Factory,
    loginViewModel: LoginViewModel,
    themeViewModel: ThemeViewModel = viewModel<ThemeViewModel>(factory = factory),
    navController: NavHostController = rememberNavController()
) {
    val backStackEntry by navController.currentBackStackEntryAsState()
    val startScreen = if (loginViewModel.connected) AppScreen.Main else AppScreen.Login
    val currentScreen = AppScreen.valueOf(
        backStackEntry?.destination?.route ?: startScreen.name
    )
    // shared with all views, no need to scope it
    val migrationViewModel = viewModel<DistribMigrationViewModel>(factory = factory)
    val mainViewModel = viewModel<MainViewModel>(factory = factory)
    LaunchedEffect(loginViewModel.connected) {
        mainViewModel.refreshAccountName()
    }

    Scaffold(
        topBar = {
            when (currentScreen) {
                AppScreen.Login -> {
                    Spacer(Modifier)
                } // Nothing
                AppScreen.Main -> {
                    MainAppBarOrSelection(
                        mainViewModel,
                        onGoToSettings = {
                            navController.navigate(AppScreen.Settings.name)
                        }
                    )
                }
                else -> null
            } ?: DefaultTopBar(
                currentScreen,
                canNavigateBack = navController.previousBackStackEntry != null,
                navigateUp = { navController.navigateUp() }
            )
        },
        // Exclude nav bar for the login view
        contentWindowInsets = WindowInsets.ime
    ) { innerPadding ->
        NavHost(
            navController = navController,
            startDestination = startScreen.name,
            modifier = Modifier
                .fillMaxSize()
                .padding(innerPadding)
                .verticalScroll(rememberScrollState())
        ) {
            composable(route = AppScreen.Login.name) {
                val vm = viewModel<StartViewModel>(factory = factory)
                StartScreen(
                    vm,
                    onLogin = {
                        loginViewModel.onLogin()
                    }
                )
            }
            composable(
                route = AppScreen.Main.name,
                exitTransition = {
                    when (targetState.destination.route) {
                        AppScreen.Settings.name -> slideOutFrom(Dir.Right)
                        else -> fadeOut()
                    }
                },
                popEnterTransition = {
                    when (initialState.destination.route) {
                        AppScreen.Settings.name -> slideInTo(Dir.Right)
                        else -> fadeIn()
                    }
                }
            ) {
                MainScreen(
                    mainViewModel,
                    migrationViewModel,
                    Modifier
                        .windowInsetsPadding(WindowInsets.safeDrawing)
                )
            }
            composable(
                route = AppScreen.Settings.name,
                enterTransition = { slideInTo(Dir.Left) },
                popExitTransition = { slideOutFrom(Dir.Left) }
            ) {
                val vm = viewModel<SettingsViewModel>(factory = factory)
                SettingsScreen(
                    vm,
                    themeViewModel,
                    loginViewModel,
                    migrationViewModel
                )
            }
        }
    }
}

@SuppressLint("ViewModelConstructorInComposable")
@Preview
@Composable
fun PreviewMain() = App(
    factory = PreviewFactory(LocalContext.current),
    LoginViewModel(true)
)

@SuppressLint("ViewModelConstructorInComposable")
@Preview
@Composable
fun PreviewStart() = App(
    factory = PreviewFactory(LocalContext.current),
    LoginViewModel(false)
)
