package org.unifiedpush.distributor.nextpush.activities

import android.app.Application
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.launch
import org.unifiedpush.android.distributor.ui.compose.BatteryOptimisationViewModel
import org.unifiedpush.android.distributor.ui.compose.RegistrationsViewModel
import org.unifiedpush.android.distributor.ui.compose.state.RegistrationListState
import org.unifiedpush.distributor.nextpush.account.AccountFactory
import org.unifiedpush.distributor.nextpush.activities.ui.MainUiState

class MainViewModel(
    mainUiState: MainUiState,
    val batteryOptimisationViewModel: BatteryOptimisationViewModel,
    val registrationsViewModel: RegistrationsViewModel,
    val application: Application? = null
) : ViewModel() {
    constructor(application: Application) : this(
        mainUiState = MainUiState(application),
        batteryOptimisationViewModel = BatteryOptimisationViewModel(application),
        registrationsViewModel = RegistrationsViewModel(
            getRegistrationListState(application)
        ),
        application
    )

    var mainUiState by mutableStateOf(mainUiState)
        private set

    private var lastDebugClickTime by mutableLongStateOf(0L)

    private var debugClickCount by mutableIntStateOf(0)

    fun refreshRegistrations() {
        viewModelScope.launch {
            application?.let {
                registrationsViewModel.state = getRegistrationListState(it)
            }
        }
    }

    fun refreshAccountName() {
        viewModelScope.launch {
            application?.let {
                mainUiState = mainUiState.copy(accountName = AccountFactory.getAccount(it)?.name ?: "")
            }
        }
    }

    fun deleteSelection() {
        viewModelScope.launch {
            val state = registrationsViewModel.state
            val tokenList = state.list.filter { it.selected }.map { it.token }
            publishAction(
                AppAction(AppAction.Type.DeleteRegistration(tokenList))
            )
            registrationsViewModel.state = RegistrationListState(
                list = state.list.filter {
                    !it.selected
                },
                selectionCount = 0
            )
        }
    }

    fun restartService() {
        viewModelScope.launch {
            publishAction(AppAction(AppAction.Type.RestartService))
        }
    }

    fun addNotificationChannel(title: String) {
        viewModelScope.launch {
            publishAction(
                AppAction(AppAction.Type.AddChannel(title))
            )
        }
    }

    fun addDebugClick() {
        val currentTime = System.currentTimeMillis()
        if (currentTime - lastDebugClickTime < 500) {
            debugClickCount++
            if (debugClickCount == 5) {
                mainUiState = mainUiState.copy(showDebugInfo = true)
            }
        } else {
            debugClickCount = 1
        }
        lastDebugClickTime = currentTime
    }

    fun dismissDebugInfo() {
        mainUiState = mainUiState.copy(showDebugInfo = false)
    }
}
