package org.unifiedpush.distributor.nextpush.activities

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.lifecycle.viewmodel.compose.viewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import org.unifiedpush.distributor.nextpush.Distributor
import org.unifiedpush.distributor.nextpush.EventBus
import org.unifiedpush.distributor.nextpush.account.AccountFactory
import org.unifiedpush.distributor.nextpush.activities.ui.App
import org.unifiedpush.distributor.nextpush.activities.ui.theme.AppTheme
import org.unifiedpush.distributor.nextpush.services.RestartWorker
import org.unifiedpush.distributor.nextpush.utils.TAG

class MainActivity : ComponentActivity() {
    private var jobs: MutableList<Job> = emptyList<Job>().toMutableList()

    private var onResult: (
        (
            activity: Activity,
            requestCode: Int,
            resultCode: Int,
            data: Intent?,
            block: (success: Boolean) -> Unit
        ) -> Unit
    )? = null

    var loginViewModel: LoginViewModel? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        RestartWorker.startPeriodic(this)

        enableEdgeToEdge()
        val loginVM = LoginViewModel(this.application) {
            login()
        }
        loginViewModel = loginVM
        setContent {
            val factory = ViewModelFactory(this.application)
            val themeViewModel = viewModel<ThemeViewModel>(factory = factory)
            AppTheme(
                dynamicColor = themeViewModel.dynamicColors
            ) {
                App(
                    factory,
                    loginVM,
                    themeViewModel
                )
            }
            subscribeActions()
        }
    }

    private fun subscribeActions() {
        Log.d(TAG, "Subscribing to actions")
        jobs += CoroutineScope(Dispatchers.IO).launch {
            EventBus.subscribe<AppAction> { it.handle(this@MainActivity) }
        }
    }

    override fun onDestroy() {
        Log.d(TAG, "Destroy")
        jobs.removeAll {
            it.cancel()
            true
        }
        super.onDestroy()
    }

    public override fun onActivityResult(
        requestCode: Int,
        resultCode: Int,
        data: Intent?
    ) {
        super.onActivityResult(requestCode, resultCode, data)
        onResult?.let {
            it(this@MainActivity, requestCode, resultCode, data) { success ->
                Log.d(TAG, "Connection succeed=$success")
                if (success) {
                    Distributor.enableComponents(this)
                    loginViewModel?.login()
                } else {
                    Toast.makeText(
                        applicationContext,
                        "Could not connect to UnifiedPush provider",
                        Toast.LENGTH_SHORT
                    ).show()
                }
                onResult = null
            }
        }
    }

    private fun login() {
        AccountFactory.getAccount(this, connected = false)?.let {
            onResult = {
                    activity: Activity,
                    i: Int,
                    i1: Int,
                    intent: Intent?,
                    block: (
                        success: Boolean
                    ) -> Unit
                ->
                it.onActivityResult(activity, i, i1, intent, block)
            }
            runOnUiThread {
                it.connect(this)
            }
        }
    }
}
