package org.unifiedpush.distributor.nextpush.utils

import android.app.Notification
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.net.Uri
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import org.unifiedpush.android.distributor.ui.R as LibR
import org.unifiedpush.distributor.AppNotification
import org.unifiedpush.distributor.nextpush.LocalNotification
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.activities.MainActivity
import org.unifiedpush.distributor.nextpush.services.MainRegistrationCounter

const val NOTIFICATION_ID_FOREGROUND = 51115
private const val NOTIFICATION_ID_WARNING = 51215
private const val NOTIFICATION_ID_NO_START = 51315
private const val NOTIFICATION_ID_LOW_KEEPALIVE = 51315
private const val NOTIFICATION_ID_NO_PING = 51515

class MainNotificationData(
    title: String,
    text: String,
    ticker: String,
    priority: Int,
    ongoing: Boolean
) : AppNotification.NotificationData(
    smallIcon = R.drawable.ic_logo,
    title = title,
    text = text,
    ticker = ticker,
    priority = priority,
    ongoing = ongoing,
    activity = MainActivity::class.java
)

private val Context.warningChannelData: AppNotification.ChannelData
    get() = AppNotification.ChannelData(
        "Warning",
        this.getString(LibR.string.warning),
        NotificationManager.IMPORTANCE_HIGH,
        this.getString(LibR.string.warning_notif_description).format(R.string.app_name)
    )

class DisconnectedNotification(context: Context) :
    AppNotification(
        context,
        Notifications.disconnectedShown,
        NOTIFICATION_ID_WARNING,
        MainNotificationData(
            context.getString(R.string.app_name),
            context.getString(LibR.string.warning_notif_content).format(
                context.getString(R.string.app_name)
            ),
            context.getString(LibR.string.warning),
            Notification.PRIORITY_HIGH,
            true
        ),
        context.warningChannelData
    )

class NoPingNotification(context: Context) :
    AppNotification(
        context,
        Notifications.noPingShown,
        NOTIFICATION_ID_NO_PING,
        MainNotificationData(
            context.getString(R.string.app_name),
            context.getString(R.string.no_ping_notif_content),
            context.getString(LibR.string.warning),
            Notification.PRIORITY_HIGH,
            false
        ),
        context.warningChannelData
    )

class NoStartNotification(context: Context) :
    AppNotification(
        context,
        Notifications.noStartShown,
        NOTIFICATION_ID_NO_START,
        MainNotificationData(
            context.getString(R.string.app_name),
            context.getString(R.string.start_error_notif_content),
            context.getString(LibR.string.warning),
            Notification.PRIORITY_HIGH,
            false
        ),
        context.warningChannelData
    )

class NoServerAppNotification(val context: Context) :
    AppNotification(
        context,
        Notifications.noServerShown,
        NOTIFICATION_ID_NO_START,
        MainNotificationData(
            context.getString(R.string.app_name),
            context.getString(R.string.no_server_app_notif_content),
            context.getString(LibR.string.warning),
            Notification.PRIORITY_HIGH,
            false
        ),
        context.warningChannelData
    ) {
    override fun create(bigText: Boolean): Notification {
        createNotificationChannel()

        val notificationIntent = Intent(Intent.ACTION_VIEW)
        notificationIntent.setData(Uri.parse("https://apps.nextcloud.com/apps/uppush"))
        return createNotification(
            PendingIntent.getActivity(context, 0, notificationIntent, PendingIntent.FLAG_IMMUTABLE),
            bigText
        )
    }
}

class LowKeepAliveNotification(context: Context, keepalive: Int) :
    AppNotification(
        context,
        Notifications.lowKeepAliveShown,
        NOTIFICATION_ID_LOW_KEEPALIVE,
        MainNotificationData(
            context.getString(R.string.app_name),
            context.getString(R.string.low_keepalive_notif_content).format(keepalive),
            context.getString(LibR.string.warning),
            Notification.PRIORITY_HIGH,
            false
        ),
        context.warningChannelData
    )

class ForegroundNotification(context: Context) :
    AppNotification(
        context,
        Notifications.ignoreShown,
        NOTIFICATION_ID_FOREGROUND,
        MainNotificationData(
            context.getString(R.string.app_name),
            if (MainRegistrationCounter.oneOrMore(context)) {
                MainRegistrationCounter.getCount(context).let {
                    context.resources.getQuantityString(LibR.plurals.foreground_notif_content_with_reg, it, it)
                }
            } else {
                context.getString(LibR.string.foreground_notif_content_no_reg)
            },
            context.getString(LibR.string.foreground_service),
            Notification.PRIORITY_LOW,
            true
        ),
        ChannelData(
            "Foreground",
            context.getString(LibR.string.foreground_service),
            NotificationManager.IMPORTANCE_LOW,
            context.getString(LibR.string.foreground_notif_description)
        )
    )

class FromPushNotification(
    context: Context,
    title: String,
    content: String
) : AppNotification(
    context,
    Notifications.ignoreShown,
    Notifications.nextNotificationId.getAndIncrement(),
    MainNotificationData(
        title,
        content,
        title,
        Notification.PRIORITY_HIGH,
        false
    ),
    ChannelData(
        LocalNotification.getNotificationChannelId(context, title),
        context.getString(R.string.list_registrations_local_title, title),
        NotificationManager.IMPORTANCE_HIGH,
        context.getString(R.string.list_registrations_local_description)
    )
)

private object Notifications {
    val nextNotificationId = AtomicInteger(50000)
    val disconnectedShown = AtomicBoolean(false)
    val noPingShown = AtomicBoolean(false)
    val noStartShown = AtomicBoolean(false)
    val noServerShown = AtomicBoolean(false)
    val lowKeepAliveShown = AtomicBoolean(false)
    val ignoreShown = AtomicBoolean(true)
}
