package org.unifiedpush.distributor.nextpush.activities.ui

import android.annotation.SuppressLint
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import org.unifiedpush.android.distributor.ui.compose.PermissionsUi
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.activities.PreviewFactory
import org.unifiedpush.distributor.nextpush.activities.StartViewModel
import org.unifiedpush.distributor.nextpush.activities.ui.theme.nextcloud

/**
 * Login screen
 *
 * @param showManualLogin for the preview
 */
@Composable
fun StartScreen(
    viewModel: StartViewModel,
    onLogin: () -> Unit = {},
    modifier: Modifier = Modifier,
    showManualLogin: Boolean = false
) {
    var showManualLogin by remember { mutableStateOf(showManualLogin) }
    var showPassword by remember { mutableStateOf(false) }
    var username by remember { mutableStateOf("") }
    var password by remember { mutableStateOf("") }
    var server by remember { mutableStateOf("https://") }
    var showPermissionDialog by remember { mutableStateOf(true) }
    LaunchedEffect(viewModel.state) {
        viewModel.state?.let {
            username = it.username
            password = it.password
            server = it.server
        }
    }

    if (showPermissionDialog) {
        PermissionsUi {
            showPermissionDialog = false
        }
    }
    Column(
        modifier = modifier
            .background(nextcloud)
            .fillMaxSize(),
        verticalArrangement = Arrangement.spacedBy(8.dp, Alignment.CenterVertically),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Image(
            painter = painterResource(R.drawable.ic_logo),
            contentDescription = "NextPush Logo"
        )
        Text(
            text = stringResource(R.string.app_name),
            style = MaterialTheme.typography.headlineSmall.copy(fontWeight = FontWeight.Bold),
            color = Color.White
        )
        Button(
            colors = ButtonDefaults.buttonColors(containerColor = Color.White),
            onClick = {
                viewModel.setLoginSSO()
                onLogin()
            }
        ) {
            Text(
                color = nextcloud,
                fontWeight = FontWeight.Bold,
                text = stringResource(R.string.button_start_sso_connection)
            )
        }
        Text(
            color = Color.White,
            fontWeight = FontWeight.Bold,
            text = stringResource(R.string.button_start_manual_login),
            modifier = Modifier.clickable {
                showManualLogin = !showManualLogin
            }
        )
        if (showManualLogin) {
            TextField(
                colors = textFieldColor(),
                value = username,
                onValueChange = { username = it },
                label = { Text(stringResource(R.string.login_hint_username)) },
                maxLines = 1
            )
            TextField(
                colors = textFieldColor(),
                value = password,
                onValueChange = { password = it },
                label = { Text(stringResource(R.string.login_hint_password)) },
                maxLines = 1,
                visualTransformation = if (showPassword) {
                    VisualTransformation.None
                } else {
                    PasswordVisualTransformation()
                },
                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Password),
                trailingIcon = {
                    IconButton(
                        onClick = { showPassword = !showPassword }
                    ) {
                        Icon(
                            painter = painterResource(
                                if (showPassword) {
                                    R.drawable.ic_visibility_24
                                } else {
                                    R.drawable.ic_visibility_off_24
                                }
                            ),
                            tint = Color.White,
                            contentDescription = if (showPassword) {
                                "Hide password"
                            } else {
                                "Show password"
                            }
                        )
                    }
                }
            )
            TextField(
                colors = textFieldColor(),
                value = server,
                onValueChange = { server = it },
                label = { Text(stringResource(R.string.login_hint_nextcloud_root_url)) },
                maxLines = 1
            )
            Button(
                colors = ButtonDefaults.buttonColors(containerColor = nextcloud),
                onClick = {
                    viewModel.setLoginFromClipboard()
                }
            ) {
                Text(
                    color = Color.White,
                    fontWeight = FontWeight.Bold,
                    text = "Fill from clipboard"
                )
            }
            Button(
                colors = ButtonDefaults.buttonColors(containerColor = Color.White),
                onClick = {
                    viewModel.setLoginDirect(
                        StartUiState(
                            username,
                            password,
                            server
                        )
                    )
                    onLogin()
                }
            ) {
                Text(
                    color = nextcloud,
                    fontWeight = FontWeight.Bold,
                    text = stringResource(R.string.login_button_sign_in)
                )
            }
        }
    }
}

@Composable
private fun textFieldColor(): TextFieldColors = TextFieldDefaults.colors(
    focusedContainerColor = nextcloud,
    unfocusedContainerColor = nextcloud,
    focusedTextColor = Color.White,
    unfocusedTextColor = Color.White,
    focusedLabelColor = Color.White,
    unfocusedLabelColor = Color.White
)

@SuppressLint("ViewModelConstructorInComposable")
@Preview
@Composable
fun PreviewStartUi() {
    val vm = viewModel<StartViewModel>(factory = PreviewFactory(LocalContext.current))
    StartScreen(vm, showManualLogin = true)
}
