package org.unifiedpush.distributor.nextpush.activities.ui

import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.MoreVert
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import org.unifiedpush.android.distributor.ui.R as LibR
import org.unifiedpush.android.distributor.ui.compose.AppBar
import org.unifiedpush.android.distributor.ui.compose.UnregisterBarUi
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.activities.MainViewModel

@Composable
fun MainAppBarOrSelection(viewModel: MainViewModel, onGoToSettings: () -> Unit) {
    if (viewModel.registrationsViewModel.state.selectionCount > 0) {
        UnregisterBarUi(
            viewModel = viewModel.registrationsViewModel,
            onDelete = { viewModel.deleteSelection() }
        )
    } else {
        MainAppBar(
            viewModel,
            onGoToSettings
        )
    }
}

@Composable
fun MainAppBar(viewModel: MainViewModel, onGoToSettings: () -> Unit) {
    var openMenu by remember { mutableStateOf(false) }
    var showNotificationDialog by remember { mutableStateOf(false) }

    AppBar(
        R.string.app_name,
        canNavigateBack = false,
        navigateUp = {},
        actions = {
            IconButton(
                onClick = {
                    openMenu = !openMenu
                }
            ) {
                Icon(
                    imageVector = Icons.Default.MoreVert,
                    contentDescription = "Actions"
                )
            }
            Dropdown(
                openMenu,
                onRestart = {
                    viewModel.restartService()
                },
                onDismiss = {
                    openMenu = false
                },
                onNewChannel = {
                    openMenu = false
                    showNotificationDialog = true
                },
                onGoToSettings = onGoToSettings
            )
            if (showNotificationDialog) {
                AddChannelDialog(
                    onDismissRequest = {
                        showNotificationDialog = false
                    },
                    onConfirmation = { title ->
                        showNotificationDialog = false
                        viewModel.addNotificationChannel(title)
                    }
                )
            }
        }
    )
}

@Composable
fun Dropdown(
    expanded: Boolean,
    onRestart: () -> Unit,
    onDismiss: () -> Unit,
    onNewChannel: () -> Unit,
    onGoToSettings: () -> Unit
) {
    DropdownMenu(
        expanded = expanded,
        onDismissRequest = onDismiss
    ) {
        DropdownMenuItem(
            onClick = onRestart,
            text = {
                Text(stringResource(LibR.string.app_dropdown_restart))
            }
        )
        DropdownMenuItem(
            onClick = {
                onNewChannel()
            },
            text = {
                Text(
                    stringResource(R.string.app_dropdown_add_notification_channel)
                )
            }
        )
        DropdownMenuItem(
            onClick = onGoToSettings,
            text = {
                Text(stringResource(LibR.string.settings))
            }
        )
    }
}

@Preview
@Composable
fun LogoutConfirmationDialog(onDismissRequest: () -> Unit = {}, onConfirmation: () -> Unit = {}) {
    AlertDialog(
        title = {
            Text(stringResource(R.string.logout))
        },
        text = { Text(stringResource(R.string.dialog_logout_content)) },
        onDismissRequest = {
            onDismissRequest()
        },
        confirmButton = {
            TextButton(
                onClick = {
                    onConfirmation()
                }
            ) {
                Text(stringResource(android.R.string.ok))
            }
        },
        dismissButton = {
            TextButton(
                onClick = {
                    onDismissRequest()
                }
            ) {
                Text(
                    stringResource(android.R.string.cancel)
                )
            }
        }
    )
}
