package org.unifiedpush.distributor.nextpush.activities

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.launch
import org.unifiedpush.distributor.nextpush.AppStore
import org.unifiedpush.distributor.nextpush.DatabaseFactory
import org.unifiedpush.distributor.nextpush.Distributor
import org.unifiedpush.distributor.nextpush.Distributor.deleteApp
import org.unifiedpush.distributor.nextpush.Distributor.deleteDevice
import org.unifiedpush.distributor.nextpush.EventBus
import org.unifiedpush.distributor.nextpush.LocalNotification
import org.unifiedpush.distributor.nextpush.account.AccountFactory
import org.unifiedpush.distributor.nextpush.services.MigrationManager
import org.unifiedpush.distributor.nextpush.services.RestartWorker
import org.unifiedpush.distributor.nextpush.services.SourceManager
import org.unifiedpush.distributor.nextpush.services.StartService
import org.unifiedpush.distributor.nextpush.utils.TAG

class AppAction(private val type: Type) {
    sealed class Type {
        data object RestartService : Type()
        data object Logout : Type()
        class ShowToasts(val enable: Boolean) : Type()
        class AddChannel(val title: String) : Type()
        class CopyEndpoint(val token: String) : Type()
        class DeleteRegistration(val registrations: List<String>) : Type()
        data object FallbackIntroShown : Type()
        class FallbackDistribSelected(val distributor: String?) : Type()
        class MigrateToDistrib(val distributor: String) : Type()
        data object ReactivateUnifiedPush : Type()
    }

    fun handle(context: Context) {
        when (type) {
            is Type.RestartService -> restartService(context)
            is Type.Logout -> logout(context)
            is Type.ShowToasts -> setShowToasts(context, type)
            is Type.AddChannel -> addChannel(context, type)
            is Type.CopyEndpoint -> copyEndpoint(context, type)
            is Type.DeleteRegistration -> deleteRegistration(context, type)
            is Type.FallbackIntroShown -> fallbackIntroShown(context)
            is Type.FallbackDistribSelected -> fallbackDistribSelected(context, type)
            is Type.MigrateToDistrib -> migrateToDistrib(context, type)
            is Type.ReactivateUnifiedPush -> reactivateUnifiedPush(context)
        }
    }

    private fun restartService(context: Context) {
        Log.d(TAG, "Restarting the Listener")
        SourceManager.clearFails()
        StartService.stopService {
            RestartWorker.run(context, delay = 0)
        }
    }

    private fun logout(context: Context) {
        deleteDevice(context) {
            StartService.stopService()
            SourceManager.clearFails()
        }
        AccountFactory.logout(context)
        AppStore(context).wipe()
        Distributor.disableComponents(context)
    }

    private fun setShowToasts(context: Context, action: Type.ShowToasts) {
        AppStore(context).showToasts = action.enable
    }

    private fun addChannel(context: Context, action: Type.AddChannel) {
        LocalNotification.createChannel(context, action.title)
    }

    private fun copyEndpoint(context: Context, action: Type.CopyEndpoint) {
        val clipboard = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        val clip: ClipData = ClipData.newPlainText(
            "Endpoint",
            DatabaseFactory.getDb(context).getEndpoint(action.token)
        )
        clipboard.setPrimaryClip(clip)
    }

    private fun deleteRegistration(context: Context, action: Type.DeleteRegistration) {
        action.registrations.forEach {
            deleteApp(context, it)
        }
    }

    private fun fallbackIntroShown(context: Context) {
        MigrationManager().setFallbackIntroShown(context)
    }

    /**
     * Save fallback service
     *
     * If fallback is disabled and we have already send TEMP_UNAVAILABLE:
     * we send the endpoint again
     */
    private fun fallbackDistribSelected(context: Context, action: Type.FallbackDistribSelected) {
        MigrationManager()
            .selectFallbackService(
                context,
                action.distributor,
                SourceManager.shouldSendFallback
            )
    }

    private fun migrateToDistrib(context: Context, action: Type.MigrateToDistrib) {
        MigrationManager().migrate(context, action.distributor)
    }

    private fun reactivateUnifiedPush(context: Context) {
        MigrationManager().reactivate(context)
    }
}

fun ViewModel.publishAction(action: AppAction) {
    viewModelScope.launch {
        EventBus.publish(action)
    }
}
