@file:Suppress("ktlint:standard:no-wildcard-imports")

package org.unifiedpush.distributor.nextpush.account

import android.app.Activity
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.net.Uri
import android.text.SpannableString
import android.text.method.LinkMovementMethod
import android.text.util.Linkify
import android.util.Log
import android.widget.TextView
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.nextcloud.android.sso.AccountImporter
import com.nextcloud.android.sso.exceptions.*
import com.nextcloud.android.sso.helper.SingleAccountHelper
import com.nextcloud.android.sso.model.SingleSignOnAccount
import com.nextcloud.android.sso.ui.UiExceptionManager
import org.unifiedpush.distributor.nextpush.R
import org.unifiedpush.distributor.nextpush.utils.TAG

class SSOAccount(val context: Context) : Account<SingleSignOnAccount> {
    override var name: String? = null
    override var url: String? = null
    override var connected = false

    init {
        try {
            SingleAccountHelper.getCurrentSingleSignOnAccount(context).let {
                name = it.name
                url = it.url
            }
            connected = true
        } catch (e: NextcloudFilesAppAccountNotFoundException) {
            Log.d(TAG, "Nextcloud application not found")
            connected = false
        } catch (e: NoCurrentAccountSelectedException) {
            Log.d(TAG, "Device is not connected")
            connected = false
        }
    }

    override fun connect(activity: Activity) {
        Log.d(TAG, "Starting ResultActivity")
        try {
            AccountImporter.pickNewAccount(activity)
        } catch (e: NextcloudFilesAppNotInstalledException) {
            nextcloudAppNotInstalledDialog(activity)
        } catch (e: AndroidGetAccountsPermissionNotGranted) {
            UiExceptionManager.showDialogForException(activity, e)
        }
    }

    override fun onActivityResult(
        activity: Activity,
        requestCode: Int,
        resultCode: Int,
        data: Intent?,
        block: (success: Boolean) -> Unit
    ) {
        try {
            AccountImporter.onActivityResult(
                requestCode,
                resultCode,
                data,
                activity
            ) { account ->
                SingleAccountHelper.commitCurrentAccount(activity.applicationContext, account.name)
                block(true)
            }
        } catch (_: Exception) {
            block(false)
        }
    }

    override fun getClient(context: Context): SingleSignOnAccount? = try {
        SingleAccountHelper.getCurrentSingleSignOnAccount(context)
    } catch (e: NextcloudFilesAppAccountNotFoundException) {
        Log.w(TAG, "Nextcloud File is not found")
        null
    } catch (e: NoCurrentAccountSelectedException) {
        Log.w(TAG, "No account selected")
        null
    }

    override fun logout(context: Context) {
        AccountImporter.clearAllAuthTokens(context)
        AccountImporter.getSharedPreferences(context)
            .edit()
            .remove("PREF_CURRENT_ACCOUNT_STRING")
            .apply()
    }

    private fun nextcloudAppNotInstalledDialog(context: Context) {
        val builder = MaterialAlertDialogBuilder(context)
        var messageContent = context.getString(R.string.message_missing_nextcloud_app)
        val installIntent =
            Intent(
                Intent.ACTION_VIEW,
                Uri.parse(context.getString(R.string.uri_market_nextcloud_app))
            )
        messageContent += if (
            installIntent.resolveActivity(context.applicationContext.packageManager) != null
        ) {
            val callback = {
                context.startActivity(
                    Intent.createChooser(
                        installIntent,
                        context.getString(R.string.market_chooser_title)
                    )
                )
            }
            builder.setPositiveButton(context.getString(R.string.install)) { _: DialogInterface, _: Int ->
                callback()
            }
            builder.setNegativeButton(context.getString(android.R.string.cancel)) { _: DialogInterface, _: Int ->
            }
            "."
        } else {
            ": " + context.getString(R.string.uri_fdroid_nextcloud_app)
        }
        val msg = SpannableString(messageContent)
        Linkify.addLinks(msg, Linkify.WEB_URLS)
        val dialog = builder.setTitle(context.getString(R.string.nextcloud_files_not_found_title))
            .setMessage(msg)
            .create()

        dialog.setOnShowListener {
            dialog.findViewById<TextView>(android.R.id.message)?.let {
                it.movementMethod = LinkMovementMethod.getInstance()
            }
        }
        dialog.show()
    }
}
