@file:Suppress("ktlint:standard:no-wildcard-imports")

package org.unifiedpush.distributor.nextpush.account

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.util.Log
import android.widget.Toast
import java.io.IOException
import java.util.concurrent.TimeUnit
import okhttp3.*
import org.unifiedpush.distributor.nextpush.activities.MainActivity
import org.unifiedpush.distributor.nextpush.api.provider.ApiProvider.Companion.API_PATH

class DirectAccount(context: Context) : Account<OkHttpClient> {

    /** [DirectAccountStore] */
    private val daStore = DirectAccountStore(context)
    override val name: String? = daStore.username
    override val url: String? = daStore.url
    override var connected = daStore.connected
        set(value) {
            daStore.connected = value
            field = value
        }

    override fun connect(activity: Activity) {
        connected = false
        val client = getClient(activity) ?: return retActivity(activity)
        val url = daStore.url ?: return retActivity(activity)

        val request = try {
            Request.Builder()
                .url("${url}$API_PATH")
                .build()
        } catch (e: IllegalArgumentException) {
            Toast.makeText(activity, "Expected URL scheme 'http' or 'https'", Toast.LENGTH_SHORT).show()
            return retActivity(activity)
        }

        val call = client.newCall(request)
        call.enqueue(object : Callback {
            private val TAG = "DirectAccountCallback"
            override fun onFailure(call: Call, e: IOException) {
                Log.e(TAG, "Could not connect", e)
                retActivity(activity)
            }

            override fun onResponse(call: Call, response: Response) {
                Log.d(TAG, "Status: ${response.code}")
                connected = response.code == 200
                response.close()
                retActivity(activity)
            }
        })
    }

    override fun onActivityResult(
        activity: Activity,
        requestCode: Int,
        resultCode: Int,
        data: Intent?,
        block: (success: Boolean) -> Unit
    ) {
        block(connected)
    }

    override fun getClient(context: Context): OkHttpClient? {
        val username = daStore.username ?: return null
        val password = daStore.password ?: return null
        return OkHttpClient.Builder()
            .connectTimeout(30, TimeUnit.SECONDS)
            .authenticator(DirectAuth(username, password))
            .followRedirects(false)
            .build()
    }

    override fun logout(context: Context) {
        connected = false
        daStore.wipe()
    }

    private fun retActivity(activity: Activity) {
        activity.runOnUiThread {
            (activity as MainActivity).onActivityResult(0, 0, null)
        }
    }

    inner class DirectAuth(private val username: String, private val password: String) : Authenticator {
        override fun authenticate(route: Route?, response: Response): Request? {
            if (responseCount(response) >= 3) {
                return null
            }
            val credential = Credentials.basic(username, password)
            return response.request.newBuilder().header("Authorization", credential).build()
        }

        /** Count number of authentication request/response */
        private fun responseCount(response: Response): Int {
            var mutResponse = response
            var result = 1
            while (mutResponse.priorResponse?.also {
                    mutResponse = it
                } != null
            ) {
                result++
            }
            return result
        }
    }

    companion object {
        fun setCredentials(
            context: Context,
            url: String?,
            username: String?,
            password: String?
        ) {
            val daStore = DirectAccountStore(context)
            daStore.url = url
            daStore.username = username
            daStore.password = password
        }
    }
}
