package org.unifiedpush.distributor.nextpush

import android.content.Context
import android.os.Build
import android.util.Log
import org.unifiedpush.distributor.ChannelCreationStatus
import org.unifiedpush.distributor.Database
import org.unifiedpush.distributor.FailedReason
import org.unifiedpush.distributor.UnifiedPushDistributor
import org.unifiedpush.distributor.nextpush.account.AccountFactory.getAccount
import org.unifiedpush.distributor.nextpush.api.Api
import org.unifiedpush.distributor.nextpush.api.provider.ApiProvider.Companion.API_PATH
import org.unifiedpush.distributor.nextpush.utils.TAG

/**
 * These functions are used to send messages to other apps
 */
object Distributor : UnifiedPushDistributor() {
    override val receiverComponentName = "org.unifiedpush.distributor.nextpush.receivers.RegisterBroadcastReceiver"

    override fun getDb(context: Context): Database = DatabaseFactory.getDb(context)

    override val localNotificationHandler = object : LocalNotificationHandler {
        override fun onMessage(
            context: Context,
            title: String?,
            message: String
        ) {
            LocalNotification.showNotification(context, title, message)
            Log.d(TAG, "Local notification shown")
        }

        override fun onUnregistered(context: Context, title: String?) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                LocalNotification.removeOsNotificationChannel(
                    context,
                    title
                )
            }
        }

        override fun onRegistered(
            context: Context,
            success: Boolean,
            title: String?
        ) {
            if (success) {
                Log.d(TAG, "Channel \"${title}\" created")
                LocalNotification.newChannelSucceeded()
            } else {
                Log.d(TAG, "An error occurred when creating channel \"${title}\"")
            }
        }
    }

    fun channelIdToEndpoint(context: Context, channelId: String): String = "${getAccount(context)?.url}${API_PATH}push/$channelId"

    /**
     * Register the application with [packageName] and [channelId] on the push server
     *
     * [finishRegistration] must be called after the registration succeed or failed
     */
    override fun backendRegisterNewChannelId(
        context: Context,
        packageName: String,
        channelId: String,
        title: String?,
        vapid: String?,
        description: String?
    ) {
        val onSuccess = fun (newChannelId: String) {
            getDb(context).updateChannelId(channelId, newChannelId)
            finishRegistration(
                context,
                ChannelCreationStatus.Ok(
                    newChannelId,
                    channelIdToEndpoint(context, newChannelId)
                )
            )
        }
        val onError = fun () {
            finishRegistration(context, ChannelCreationStatus.Error(channelId, FailedReason.INTERNAL_ERROR))
        }
        if (packageName == context.packageName) {
            Api(context)
                .apiCreateApp(
                    context.getString(R.string.list_registrations_local_title, title),
                    null,
                    onSuccess,
                    onError
                )
        } else {
            Api(context)
                .apiCreateApp(packageName, vapid, onSuccess, onError)
        }
    }

    /**
     * Update the application with [packageName] and [channelId] on the push server
     *
     * [finishRegistration] must be called after the registration succeed or failed
     */
    override fun backendUpdateChannelId(
        context: Context,
        packageName: String,
        channelId: String,
        title: String?,
        vapid: String?,
        description: String?
    ) {
        val onSuccess = fun (newChannelId: String) {
            getDb(context).updateChannelId(channelId, newChannelId)
            // Remove old record from the backend
            backendUnregisterChannelId(context, channelId)
            finishRegistration(
                context,
                ChannelCreationStatus.Ok(
                    newChannelId,
                    channelIdToEndpoint(context, newChannelId)
                )
            )
        }
        val onError = fun () {
            finishRegistration(context, ChannelCreationStatus.Error(channelId, FailedReason.INTERNAL_ERROR))
        }
        if (packageName == context.packageName) {
            Api(context).apiCreateApp(context.getString(R.string.list_registrations_local_title, title), null, onSuccess, onError)
        } else {
            Api(context).apiCreateApp(packageName, vapid, onSuccess, onError)
        }
    }

    override fun backendUnregisterChannelId(context: Context, channelId: String) {
        Api(context).apiDeleteApp(channelId) {
            deleteChannelFromServer(context, channelId)
        }
    }

    fun deleteDevice(context: Context, block: () -> Unit = {}) {
        val db = DatabaseFactory.getDb(context)
        db.listApps().forEach {
            sendUnregistered(context, it)
        }
        db.deleteAllApps()
        Api(context).apiDeleteDevice(block)
    }
}
