/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TimeslotEntity from "@/data/entity/TimeslotEntity";
import { StyleSheet, View } from "react-native";
import React from "react";
import { IconButton, useTheme } from "react-native-paper";
import TimePicker from "./TimePicker";
import { SelectList } from "react-native-dropdown-select-list";
import Ionicons from "@expo/vector-icons/Ionicons";
import { weekdays } from "@/constants/Time";

export interface TimeslotPickerProps {
  selectedTimeslot: TimeslotEntity;
  setSelectedTimeslot: (ts: TimeslotEntity) => void;
  deleteTimeslot: () => void;
}

const endBeforeStart = (ts: TimeslotEntity) =>
  ts.start_hour * 60 + ts.start_minute > ts.end_hour * 60 + ts.end_minute;

export default function TimeslotPicker({
  selectedTimeslot,
  setSelectedTimeslot,
  deleteTimeslot,
}: TimeslotPickerProps) {
  const theme = useTheme();

  return (
    <View
      style={[
        {
          backgroundColor: theme.colors.elevation.level4,
          borderBottomColor: theme.colors.outline,
        },
        styles.row,
      ]}
    >
      <View style={styles.day_picker}>
        <SelectList
          search={false}
          data={weekdays}
          defaultOption={weekdays.find(
            (p) => p.key == selectedTimeslot.day_of_week,
          )}
          setSelected={(d: string) => {
            setSelectedTimeslot({ ...selectedTimeslot, day_of_week: d });
          }}
          boxStyles={{
            backgroundColor: theme.colors.elevation.level2,
            borderColor: theme.colors.elevation.level2,
            margin: 5,
          }}
          dropdownStyles={{
            backgroundColor: theme.colors.elevation.level2,
            borderColor: theme.colors.elevation.level2,
            margin: 5,
          }}
          inputStyles={{ color: theme.colors.onBackground }}
          dropdownTextStyles={{ color: theme.colors.onBackground }}
          arrowicon={
            <Ionicons
              name="chevron-down"
              color={theme.colors.primary}
              size={15}
              style={{ marginLeft: 5 }}
            ></Ionicons>
          }
          save="key"
        />
      </View>

      <TimePicker
        selectedHour={selectedTimeslot.start_hour}
        selectedMinute={selectedTimeslot.start_minute}
        setSelectedTime={(hour: number, minute: number) => {
          const newTimeslot: TimeslotEntity = selectedTimeslot;
          newTimeslot.start_hour = hour;
          newTimeslot.start_minute = minute;
          if (endBeforeStart(newTimeslot)) {
            newTimeslot.end_hour = hour;
            newTimeslot.end_minute = minute;
          }
          setSelectedTimeslot(newTimeslot);
        }}
      />

      <TimePicker
        selectedHour={selectedTimeslot.end_hour}
        selectedMinute={selectedTimeslot.end_minute}
        setSelectedTime={(hour: number, minute: number) => {
          const newTimeslot: TimeslotEntity = selectedTimeslot;
          newTimeslot.end_hour = hour;
          newTimeslot.end_minute = minute;
          if (endBeforeStart(newTimeslot)) {
            newTimeslot.start_hour = hour;
            newTimeslot.start_minute = minute;
          }
          setSelectedTimeslot(newTimeslot);
        }}
      />

      <IconButton onPress={deleteTimeslot} icon="delete" />
    </View>
  );
}

const styles = StyleSheet.create({
  row: {
    flexDirection: "row",
    alignItems: "center",
    margin: 10,
    borderRadius: 10,
    width: "90%",
  },

  day_picker: { flex: 1 },
  delete: { padding: 20 },
});
