/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import React, { useEffect, useState } from "react";
import { getTherapistsDS, loadTherapists } from "@/utils/storage";
import { FlatList } from "react-native";
import TherapistListEntry from "@/components/TherapistListEntry";
import TherapistEntity from "@/data/entity/TherapistEntity";
import { getTherapistName } from "@/utils/misc";
import { Button, Dialog, Portal } from "react-native-paper";

export interface TherapistPickerProps {
  selected: TherapistEntity | undefined;
  setSelected: (t: TherapistEntity | undefined) => void;
}

export default function TherapistPicker({
  selected,
  setSelected,
}: TherapistPickerProps) {
  const TherapistDS = getTherapistsDS();
  const [showTherapistList, setShowTherapistList] = useState(false);
  const [therapists, setTherapists] = useState<TherapistEntity[]>([]);

  useEffect(() => {
    loadTherapists(TherapistDS).then(setTherapists);
  }, []);

  return (
    <>
      <Button
        onPress={() => {
          setShowTherapistList(true);
        }}
      >
        {selected ? getTherapistName(selected) : "Therapeut*in auswählen"}
      </Button>
      <Portal>
        <Dialog
          visible={showTherapistList}
          dismissableBackButton={true}
          dismissable={true}
          onDismiss={() => {
            setShowTherapistList(false);
          }}
        >
          <Dialog.Title>Therapeut*in auswählen</Dialog.Title>
          <Dialog.ScrollArea>
            <FlatList
              data={therapists}
              renderItem={(item) => (
                <TherapistListEntry
                  name={getTherapistName(item.item)}
                  onPress={() => {
                    setSelected(item.item);
                    setShowTherapistList(false);
                  }}
                />
              )}
            />
          </Dialog.ScrollArea>
        </Dialog>
      </Portal>
    </>
  );
}
