/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import { Button, Dialog, Portal, Text, useTheme } from "react-native-paper";
import ProtocolEntity from "@/data/entity/ProtocolEntity";
import { getTherapistName } from "@/utils/misc";
import { TouchableOpacity, View, StyleSheet } from "react-native";
import { CallResultLabels } from "@/constants/Protocol";
import { useState } from "react";
import { deleteProtocolEntry, getTherapistsDS } from "@/utils/storage";

interface ProtocolListEntryProps {
  entry: ProtocolEntity;
  onChange?: () => void;
  short?: boolean;
}

export default function ProtocolListEntry({
  entry,
  onChange,
  short,
}: ProtocolListEntryProps) {
  const TherapistDS = getTherapistsDS();
  const theme = useTheme();
  const [showDialog, setShowDialog] = useState(false);

  return (
    <>
      <TouchableOpacity
        onPress={() => {
          setShowDialog(true);
        }}
      >
        <View
          style={{
            backgroundColor: theme.colors.background,
            ...styles.container,
          }}
        >
          <View style={styles.row}>
            <View style={styles.textContainer}>
              <Text style={styles.bottomText}>
                {new Date(entry.createdDate).toLocaleString()}
              </Text>
              {!short && (
                <Text style={styles.nameText}>
                  {getTherapistName(entry.therapist)}
                </Text>
              )}
              <Text style={styles.bottomText}>
                {CallResultLabels[entry.callResult].value}
                {!short && entry.remark && "\n" + entry.remark}
              </Text>
            </View>
          </View>
        </View>
      </TouchableOpacity>
      <Portal>
        <Dialog
          visible={showDialog}
          dismissable={true}
          dismissableBackButton={true}
          onDismiss={() => {
            setShowDialog(false);
          }}
        >
          <Dialog.ScrollArea>
            <Text style={styles.bottomText}>
              {new Date(entry.createdDate).toLocaleString()}
            </Text>
            <Text style={styles.nameText}>
              {getTherapistName(entry.therapist)}
            </Text>
            <Text style={styles.bottomText}>
              {CallResultLabels[entry.callResult].value}
              {entry.remark && "\n" + entry.remark}
            </Text>
          </Dialog.ScrollArea>
          <Dialog.Actions>
            <Button
              onPress={() => {
                deleteProtocolEntry(TherapistDS, entry.id);
                setShowDialog(false);
                if (onChange) onChange();
              }}
            >
              Delete
            </Button>
          </Dialog.Actions>
        </Dialog>
      </Portal>
    </>
  );
}

const styles = StyleSheet.create({
  container: {
    width: "100%",
    padding: 3,
  },
  row: {
    flexDirection: "row",
  },
  textContainer: {
    justifyContent: "center",
    maxWidth: "80%",
    flexGrow: 1,
    minHeight: 60,
    padding: 5,
  },
  nameText: {
    fontSize: 18,
  },
  bottomText: {
    fontSize: 15,
  },
});
