/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import notifee, {
  AndroidNotificationSetting,
  AuthorizationStatus,
  RepeatFrequency,
  TimestampTrigger,
  TriggerType,
} from "@notifee/react-native";
import { ReactNode, useEffect } from "react";
import { getTherapistsDS, loadEnabledTimeslots } from "@/utils/storage";
import { ts_end, ts_start } from "@/utils/reachability";
import { weekdays } from "@/constants/Time";
import { Platform } from "react-native";

interface TimeslotCount {
  start: number;
  count: number;
}

export const NotifcationScheduler = ({ children }: { children: ReactNode }) => {
  const TherapistDS = getTherapistsDS();

  const scheduleNotifications = async () => {
    const settings = await notifee.getNotificationSettings();
    if (
      settings.android.alarm == AndroidNotificationSetting.ENABLED &&
      settings.authorizationStatus > AuthorizationStatus.DENIED
    ) {
      await notifee.createChannel({
        id: "therapistCount",
        name: "Erreichbare Therapeut*innen",
      });

      const enabledTimeslots = await loadEnabledTimeslots(TherapistDS);
      // timeslotCounts per weekday
      const timeslotCounts = weekdays.map((_, wd) => {
        const timeslots = enabledTimeslots.filter(
          (ts) => ts.day_of_week === weekdays[wd].key,
        );
        const times = new Set(timeslots.map(ts_start));
        const endTimes = new Set(timeslots.map(ts_end));
        endTimes.forEach((v) => times.add(v)); // this is equal to `startTimes = startTimes.union(endTimes)` except for that the latter doesn't actually return. I'm so sick and tired of this fucking language.

        const timeslotCount = Array.from(times)
          .sort()
          .map((v) => new Object({ start: v, count: 0 }) as TimeslotCount);

        timeslots.forEach((ts) => {
          const start = timeslotCount.findIndex(
            (c) => c.start === ts_start(ts),
          );
          const end = timeslotCount.findLastIndex((c) => c.start < ts_end(ts));
          timeslotCount.slice(start, end + 1).forEach((c) => c.count++);
        });

        return { wd: wd, tsc: timeslotCount };
      });

      const now = new Date(Date.now());
      const notifications = Array.from(
        timeslotCounts.map(({ wd, tsc }) =>
          Array.from(
            tsc
              .filter(({ count }) => count != 0)
              .map(({ start, count }) => {
                // FIXME: replace some of the following code by nextStartingDates
                const triggerTime = new Date(Date.now());
                triggerTime.setHours(start / 60);
                triggerTime.setMinutes(start % 60);
                triggerTime.setSeconds(0);
                triggerTime.setDate(
                  now.getDate() + ((7 + wd - now.getDay()) % 7),
                );
                if (triggerTime.getTime() < now.getTime() + 1000 * 60)
                  triggerTime.setDate(triggerTime.getDate() + 7); // if triggerTime not at least a minute in the future: schedule for next week

                const notification: Notification = {
                  id: `therapist_count_${start.toString()}`,
                  title:
                    count == 1
                      ? `Ein*e Therapeut*in ist erreichbar.`
                      : `${count.toString()} Therapeut*innen sind erreichbar.`,
                  android: {
                    channelId: "therapistCount",
                    pressAction: {
                      id: "default",
                    },
                  },
                };
                const trigger: TimestampTrigger = {
                  alarmManager: true, //FIXME: how reliable are the notifications without alarmManager?
                  type: TriggerType.TIMESTAMP,
                  timestamp: triggerTime.getTime(),
                  repeatFrequency: RepeatFrequency.WEEKLY,
                };
                return { notification: notification, trigger: trigger };
              }),
          ),
        ),
      )
        .flat()
        .sort((a, b) => a.trigger.timestamp - b.trigger.timestamp);

      await notifee.cancelTriggerNotifications();

      const NOTIFICATION_TRIGGER_LIMIT = Platform.OS == "android" ? 50 : 64;
      await Promise.all(
        notifications
          .slice(0, NOTIFICATION_TRIGGER_LIMIT)
          .map(async ({ notification, trigger }) =>
            notifee.createTriggerNotification(notification, trigger),
          ),
      );
    }
  };

  useEffect(() => {
    scheduleNotifications();
  }, []);

  return <>{children}</>;
};
