/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import { Alert, StyleSheet, TouchableOpacity } from "react-native";
import { Text, useTheme } from "react-native-paper";
import Ionicons from "@expo/vector-icons/Ionicons";
import { Linking, Platform } from "react-native";

export interface ContactButtonProps {
  kind: "email" | "phone" | "mobile";
  destination: string;
}

export default function ContactButton(props: ContactButtonProps) {
  const theme = useTheme();

  const getIcon = () => {
    switch (props.kind) {
      case "email":
        return "at";
      case "phone":
        return "call";
      case "mobile":
        return "phone-portrait";
      default:
        return "help-outline";
    }
  };

  const getTitle = () => {
    switch (props.kind) {
      case "email":
        return "E-Mail";
      case "phone":
      case "mobile":
        return "Anrufen";
    }
  };

  const openDestination = () => {
    let url = "";
    if (props.kind == "email") {
      url = `mailto:${props.destination}`;
    } else {
      if (Platform.OS !== "android") {
        url = `telprompt:${props.destination}`;
      } else {
        url = `tel:${props.destination}`;
      }
    }
    Linking.canOpenURL(url)
      .then((supported) => {
        if (!supported) {
          Alert.alert(`Kontaktmöglichkeit nicht verfügbar: "${url}"`);
        } else {
          return Linking.openURL(url);
        }
      })
      .catch((err: unknown) => {
        console.log(err);
      });
  };

  return (
    <TouchableOpacity
      style={{
        borderColor: theme.colors.elevation.level5,
        backgroundColor: theme.colors.elevation.level4,
        ...styles.contactButton,
      }}
      onPress={openDestination}
    >
      <Text style={{ color: theme.colors.onBackground, ...styles.titleText }}>
        {getTitle()}
      </Text>
      <Ionicons
        name={getIcon()}
        color={theme.colors.primary}
        size={50}
      ></Ionicons>
    </TouchableOpacity>
  );
}

const styles = StyleSheet.create({
  contactButton: {
    minWidth: 150,
    maxWidth: 150,
    minHeight: 150,
    maxHeight: 150,
    borderRadius: 25,
    borderWidth: 2,
    alignContent: "center",
    justifyContent: "center",
    alignItems: "center",
    margin: 10,
    padding: 10,
  },
  titleText: {
    fontSize: 20,
    marginBottom: 15,
  },
});
