/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import ContactButton from "@/components/ContactButton";
import ProtocolForm from "@/components/ProtocolForm";
import { Button, Text } from "react-native-paper";
import TherapistEntity from "@/data/entity/TherapistEntity";
import { getPrintableAddress, getTherapistName } from "@/utils/misc";
import { getTherapistsDS, loadTherapist } from "@/utils/storage";
import Ionicons from "@expo/vector-icons/Ionicons";
import { useLocalSearchParams, useNavigation, useRouter } from "expo-router";
import React, { useEffect, useState } from "react";
import {
  View,
  StyleSheet,
  ScrollView,
  TouchableWithoutFeedback,
  Keyboard,
  KeyboardAvoidingView,
  TouchableOpacity,
  Platform,
  Linking,
} from "react-native";
import { useHeaderHeight } from "@react-navigation/elements";
import Toast from "react-native-toast-message";
import ProtocolListEntry from "@/components/ProtocolListEntry";

export default function CallTherapist() {
  const navigation = useNavigation();
  const router = useRouter();
  const headerHeight = Platform.OS === "android" ? 0 : useHeaderHeight();
  const TherapistsDS = getTherapistsDS();

  const { id } = useLocalSearchParams<{ id: string }>();
  const [therapist, setTherapist] = useState<TherapistEntity | null>(null);
  const [showNewProtocol, setShowNewProtocol] = useState(false);

  const reloadTherapist = () => {
    loadTherapist(TherapistsDS, id)
      .then(setTherapist)
      .catch((e: unknown) => {
        console.error(`Failed loading therapist with ID ${id}:`, e);
      });
  };

  useEffect(() => {
    reloadTherapist();
    navigation.setOptions({
      headerRight: () => (
        // onPressIn is used until this is fixed: https://github.com/expo/expo/issues/33093
        <Button
          onPressIn={() => {
            router.push({
              pathname: `/list/edit`,
              params: { id: id },
            });
          }}
        >
          <Ionicons name="open-outline" size={24} />
        </Button>
      ),
    });
  }, []);

  useEffect(() => {
    if (therapist !== null) {
      navigation.setOptions({
        title: getTherapistName(therapist),
      });
    }
  }, [therapist]);
  return (
    <>
      {therapist ? (
        <ScrollView>
          <KeyboardAvoidingView
            behavior="position"
            style={{ flex: 1, paddingTop: headerHeight }}
          >
            <TouchableWithoutFeedback onPress={Keyboard.dismiss}>
              <View style={styles.container}>
                <View style={styles.contactButtons}>
                  {!!therapist.phone && (
                    <ContactButton
                      kind="phone"
                      destination={therapist.phone}
                    ></ContactButton>
                  )}
                  {!!therapist.mobile && (
                    <ContactButton
                      kind="mobile"
                      destination={therapist.mobile}
                    ></ContactButton>
                  )}
                  {!!therapist.email && (
                    <ContactButton
                      kind="email"
                      destination={therapist.email}
                    ></ContactButton>
                  )}
                </View>
                <View style={styles.addressBox}>
                  <TouchableOpacity
                    onPress={() => {
                      if (!!therapist.lat && !!therapist.lon) {
                        const scheme = Platform.select({
                          ios: "maps://0,0?q=",
                          android: "geo:0,0?q=",
                        });
                        const label = getTherapistName(therapist);
                        /* eslint-disable @typescript-eslint/restrict-template-expressions */
                        const url =
                          Platform.select({
                            ios: `${scheme}${label}@${therapist.lat},${therapist.lon}`,
                            android: `${scheme}${therapist.lat},${therapist.lon}(${label})`,
                          }) ?? "";
                        /* eslint-enable @typescript-eslint/restrict-template-expressions */
                        Linking.canOpenURL(url).then(() =>
                          Linking.openURL(url),
                        );
                      }
                    }}
                  >
                    <Text>{getPrintableAddress(therapist)}</Text>
                  </TouchableOpacity>
                </View>
                <View style={styles.protocolBox}>
                  <Text
                    style={{
                      fontSize: 20,
                      fontWeight: "bold",
                      marginBottom: 10,
                    }}
                  >
                    Protokolleinträge
                  </Text>
                  {therapist.protocolEntries.map((entry) => (
                    <ProtocolListEntry
                      key={entry.id}
                      entry={{ ...entry, therapist: therapist }}
                      onChange={() => {
                        reloadTherapist();
                      }}
                      short={true}
                    />
                  ))}
                  <Button
                    onPress={() => {
                      setShowNewProtocol(true);
                    }}
                  >
                    Protolleintrag hinzufügen
                  </Button>
                  <ProtocolForm
                    visible={showNewProtocol}
                    setVisible={setShowNewProtocol}
                    target={therapist}
                    onSave={() => {
                      reloadTherapist();
                      Toast.show({
                        type: "success",
                        text1: "Protokolleintrag wurde gespeichert",
                      });
                    }}
                  />
                </View>
              </View>
            </TouchableWithoutFeedback>
          </KeyboardAvoidingView>
        </ScrollView>
      ) : (
        <></>
      )}
      <Toast position="bottom" bottomOffset={20} />
    </>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1 },
  contactButtons: {
    paddingHorizontal: 25,
    marginTop: 20,
    justifyContent: "center",
    flexDirection: "row",
    paddingBottom: 20,
    borderBottomStartRadius: 40,
    borderBottomEndRadius: 40,
  },
  addressBox: { paddingHorizontal: 25 },
  protocolBox: { paddingHorizontal: 25, marginTop: 40 },
});
