/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import "reflect-metadata"; // needed for typeorm
import {
  NavThemeDark,
  NavThemeLight,
  TFDarkTheme,
  TFLightTheme,
} from "@/constants/Colors";
import { ThemeProvider as NavigationThemeProvider } from "@react-navigation/native";
import { Stack } from "expo-router/stack";
import { MenuProvider } from "react-native-popup-menu";
import { DataProvider } from "@/components/DataProvider";
import { Platform, useColorScheme } from "react-native";
import { BlurView } from "expo-blur";
import { PaperProvider } from "react-native-paper";
import { NotifcationScheduler } from "@/components/NotifcationScheduler";
import { useEffect, useState } from "react";
import { loadSettings, Settings, storeSettings } from "@/utils/settings";
import Welcome from "@/app/welcome";
import { StatusBar } from "expo-status-bar";

export default function Layout() {
  const scheme = useColorScheme() === "dark";
  const paperTheme = scheme ? TFDarkTheme : TFLightTheme;
  const [showWelcome, setShowWelcome] = useState(false);

  useEffect(() => {
    loadSettings().then((value: Settings) => {
      setShowWelcome(value.showWelcome);
    });
  }, []);

  return (
    <PaperProvider theme={paperTheme}>
      <NavigationThemeProvider value={scheme ? NavThemeDark : NavThemeLight}>
        <DataProvider>
          <NotifcationScheduler>
            <MenuProvider>
              <StatusBar
                translucent={false}
                backgroundColor={paperTheme.colors.background}
              />
              {showWelcome ? (
                <Welcome
                  dismiss={() => {
                    setShowWelcome(false);
                    storeSettings({ showWelcome: false });
                  }}
                />
              ) : (
                <Stack
                  screenOptions={{ headerBackButtonDisplayMode: "minimal" }}
                >
                  <Stack.Screen
                    name="(tabs)"
                    options={{ headerShown: false }}
                  />
                  <Stack.Screen
                    name="call/therapist"
                    options={{
                      title: "Anrufen",
                      headerTransparent: Platform.OS !== "android",
                      headerStyle: {
                        backgroundColor:
                          Platform.OS === "android"
                            ? paperTheme.colors.background
                            : "transparent",
                      },
                      headerBackground: () => (
                        <BlurView intensity={50} style={{ flex: 1 }} />
                      ),
                    }}
                  />
                  <Stack.Screen
                    name="list/edit"
                    options={{
                      title: "Bearbeiten",
                      headerTransparent: Platform.OS !== "android",
                      headerStyle: {
                        backgroundColor:
                          Platform.OS === "android"
                            ? paperTheme.colors.background
                            : "transparent",
                      },
                      headerBackground: () => (
                        <BlurView intensity={50} style={{ flex: 1 }} />
                      ),
                    }}
                  />
                  <Stack.Screen
                    name="list/settings"
                    options={{
                      title: "Einstellungen",
                    }}
                  />
                </Stack>
              )}
            </MenuProvider>
          </NotifcationScheduler>
        </DataProvider>
      </NavigationThemeProvider>
    </PaperProvider>
  );
}
