/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TherapistEntity from "@/data/entity/TherapistEntity";
import { DataContext } from "@/components/DataProvider";
import { useContext } from "react";
import { shareAsync } from "expo-sharing";
import * as FileSystem from "expo-file-system";
import { DataSource, DeleteResult } from "typeorm";
import TimeslotEntity from "@/data/entity/TimeslotEntity";
import ProtocolEntity from "@/data/entity/ProtocolEntity";
import * as SQLite from "expo-sqlite";
import { getDocumentAsync } from "expo-document-picker";

export function getTherapistsDS(): DataSource {
  return useContext(DataContext);
}

export function getTherapistsDSLoader() {
  return useContext(DataContext);
}

export async function loadTherapists(
  TherapistsDS: DataSource,
): Promise<TherapistEntity[]> {
  return TherapistsDS.manager.find(TherapistEntity, {
    relations: { timeslots: true },
  });
}

export async function loadTherapist(
  TherapistsDS: DataSource,
  id: number,
): Promise<TherapistEntity | null> {
  return TherapistsDS.manager.findOne(TherapistEntity, {
    where: { id: id },
    relations: { timeslots: true, protocolEntries: true },
  });
}

export async function storeTherapist(
  TherapistsDS: DataSource,
  t: TherapistEntity,
): Promise<TherapistEntity> {
  return TherapistsDS.manager.save(TherapistEntity, t);
}

export async function storeTherapists(
  TherapistsDS: DataSource,
  ts: TherapistEntity[],
) {
  return TherapistsDS.manager.save(TherapistEntity, ts);
}

export async function deleteTherapist(
  TherapistsDS: DataSource,
  id: number,
): Promise<DeleteResult> {
  await TherapistsDS.manager.delete(TimeslotEntity, { therapist: id });
  return TherapistsDS.manager.delete(TherapistEntity, id);
}

export async function createTherapist(
  TherapistsDS: DataSource,
): Promise<TherapistEntity> {
  const t = TherapistsDS.manager.create(TherapistEntity, { enabled: true });
  return TherapistsDS.manager.save(TherapistEntity, t);
}

export async function loadTimeslots(
  TherapistsDS: DataSource,
): Promise<TimeslotEntity[]> {
  return TherapistsDS.manager.find(TimeslotEntity, {
    relations: { therapist: true },
  });
}

export async function loadEnabledTimeslots(TherapistsDS: DataSource) {
  const timeslots = await loadTimeslots(TherapistsDS);
  return timeslots.filter((ts) => ts.therapist.enabled);
}

export async function createProtocolEntry(
  TherapistsDS: DataSource,
  callResult: number,
  remark: string | null,
  therapist: TherapistEntity,
): Promise<ProtocolEntity> {
  const pe = TherapistsDS.manager.create(ProtocolEntity, {
    callResult: callResult,
    remark: remark,
    therapist: therapist,
  });
  return TherapistsDS.manager.save(ProtocolEntity, pe);
}

export async function deleteProtocolEntry(
  TherapistsDS: DataSource,
  id: number,
): Promise<DeleteResult> {
  return TherapistsDS.manager.delete(ProtocolEntity, id);
}

export async function loadProtocolEntries(
  TherapistsDS: DataSource,
): Promise<ProtocolEntity[]> {
  return TherapistsDS.manager.find(ProtocolEntity, {
    relations: { therapist: true },
    order: { createdDate: "DESC" },
  });
}

export async function backupDatabase(TherapistsDS: DataSource) {
  const dbName = TherapistsDS.options.database as string;
  const dbBackupName = dbName + ".bak";

  const [db, dbBackup] = await Promise.all([
    SQLite.openDatabaseAsync(dbName),
    SQLite.openDatabaseAsync(dbBackupName),
  ]);
  await SQLite.backupDatabaseAsync({
    sourceDatabase: db,
    destDatabase: dbBackup,
  });
  await Promise.all([db.closeAsync(), dbBackup.closeAsync()]);

  await shareAsync("file://" + dbBackup.databasePath);

  await SQLite.deleteDatabaseAsync(dbBackupName);

  await TherapistsDS.destroy(); // FIXME: this should not be necessary, but the DS seems to loose access without reinitialization
  await TherapistsDS.initialize();
}

export async function restoreDatabase(TherapistsDS: DataSource) {
  const documentPickerResult = await getDocumentAsync();
  if (documentPickerResult.canceled) return;

  const dbName = TherapistsDS.options.database as string;
  const dbBackupName = dbName + ".bak";

  await FileSystem.moveAsync({
    from: documentPickerResult.assets[0].uri,
    // eslint-disable-next-line @typescript-eslint/restrict-plus-operands
    to: "file://" + SQLite.defaultDatabaseDirectory + "/" + dbBackupName,
  });

  const [db, dbBackup] = await Promise.all([
    SQLite.openDatabaseAsync(dbName),
    SQLite.openDatabaseAsync(dbBackupName),
  ]);
  await SQLite.backupDatabaseAsync({
    sourceDatabase: dbBackup,
    destDatabase: db,
  });
  await Promise.all([db.closeAsync(), dbBackup.closeAsync()]);

  await SQLite.deleteDatabaseAsync(dbBackupName);

  await TherapistsDS.destroy(); // FIXME: this should not be necessary, but the DS seems to loose access without reinitialization
  await TherapistsDS.initialize();
}
