/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TimeslotEntity from "@/data/entity/TimeslotEntity";
import { getDayOfWeekAsString, partitionArray } from "./misc";

export function ts_start(ts: TimeslotEntity) {
  return ts.start_hour * 60 + ts.start_minute;
}

export function ts_end(ts: TimeslotEntity) {
  return ts.end_hour * 60 + ts.end_minute;
}

interface Reachabilities {
  currently: TimeslotEntity[];
  soon: TimeslotEntity[];
  later: TimeslotEntity[][];
}

/**
 * Returns a object containing currently, soon and later available
 * timeslots based on the following cases/criteria:
 *
 * let n = new Date();
 * now = n.hour * 60 + n.minute;
 *
 * a) currently reachable:
 * now is between start and end of the timeslot
 * >>> start <= now && end > now
 *
 * b) soon reachable:
 * start is later than now and start is earlier than the end of this hour
 * >>> start > now && start < (n.hour + 1) * 60
 *
 * c) later reachable (hour x after now):
 * start is later or equal than the first minute of the x hour, the
 * end doesn't matter
 * >>> start >= (n.hour + x) * 60 && start < (n.hour + x + 1) * 60
 *
 * c+x) exludes c) excludes b) excludes a)
 *
 */
export function getReachabilities(
  all_timeslots: TimeslotEntity[],
  hours: number,
) {
  const result: Reachabilities = {
    currently: [],
    soon: [],
    later: [],
  };

  const d_now = new Date();
  const hour = d_now.getHours();
  const minute = d_now.getMinutes();
  const now = hour * 60 + minute;

  all_timeslots = all_timeslots.filter(
    (ts) => ts.day_of_week == getDayOfWeekAsString(d_now.getDay()),
  );

  // a) currently
  const currently_filter_result = partitionArray(
    all_timeslots,
    (ts) => ts_start(ts) <= now && ts_end(ts) > now,
  );
  result.currently = currently_filter_result.right; // store according therapists
  all_timeslots = currently_filter_result.wrong;

  // b) soon
  const soon_filter_result = partitionArray(
    all_timeslots,
    (ts) => ts_start(ts) > now && ts_start(ts) < (hour + 1) * 60,
  );
  result.soon = soon_filter_result.right;
  all_timeslots = soon_filter_result.wrong;

  // c) later + x
  for (let x = 1; x < hours + 1; x++) {
    const later_filter_result = partitionArray(
      all_timeslots,
      (ts) =>
        ts_start(ts) > (hour + x) * 60 && ts_start(ts) < (hour + x + 1) * 60,
    );
    result.later.push(later_filter_result.right);
    all_timeslots = later_filter_result.wrong;
  }

  return result;
}
