/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import * as Print from "expo-print";
import { shareAsync } from "expo-sharing";

import * as nj from "nunjucks";
import { loadProtocolEntries } from "./storage";
import { DataSource } from "typeorm";
import { getAddressWithoutName, getTherapistName } from "./misc";
import { CallResultLabels } from "@/constants/Protocol";

const template = `<html>
  <head>
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, minimum-scale=1.0, user-scalable=no" />
    <style>
      body {
        font-family: Verdana, sans-serif;
        margin: 1cm;
      }

      h3 {
        margin-bottom: 20px;
        font-size: 16pt;
      }

      hr {
        border: 1px solid #1c8bd7;
        margin: 20px 0;
      }

      table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
      }

      th, td {
        border: 1px solid #ddd;
        padding: 8px;
        text-align: left;
        font-size: 8pt;
      }
    </style>
  </head>
  <body>
    <h3>
      Therapiefinder - Protokoll Export vom {{ export_date }}
    </h3>
    <hr/>
    <table>
      <tr>
        <th>#</th>
        <th>Zeitpunkt</th>
        <th>Name</th>
        <th>Niederlassung</th>
        <th>Ergebnis des Anrufs</th>
        <th>Bemerkung</th>
      </tr>
      {% for entry in protocol_entries %}
        <tr class="row">
          <td>{{ entry.id }}</td>
          <td>{{ entry.createdDate }}</td>
          <td>{{ entry.name }}</td>
          <td>{{ entry.address }}</td>
          <td>{{ entry.result }}</td>
          <td>{{ entry.remark }}</td>
        </tr>
      {% endfor %}
    </table>
  </body>
</html>
`;

export default async function exportProtocolAsPDF(therapistDS: DataSource) {
  nj.configure("views", { autoescape: true });

  const protocol_entries = (
    await loadProtocolEntries(therapistDS)
  ).toReversed();

  const context = {
    export_date: new Date().toLocaleDateString(),
    protocol_entries: protocol_entries.map((e) => ({
      ...e,
      createdDate:
        e.createdDate.toLocaleDateString() +
        " um " +
        e.createdDate.toLocaleTimeString(),
      name: getTherapistName(e.therapist),
      address: getAddressWithoutName(e.therapist),
      result: CallResultLabels.find((p) => p.key.valueOf() === e.callResult)
        ?.value,
    })),
  };
  const html = nj.renderString(template, context);

  const { uri } = await Print.printToFileAsync({
    html: html,
    height: 595,
    width: 840,
  });
  console.log("File has been saved to:", uri);
  await shareAsync(uri, { UTI: ".pdf", mimeType: "application/pdf" });
}
