/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TherapistEntity from "@/data/entity/TherapistEntity";
import TimeslotEntity from "@/data/entity/TimeslotEntity";

export enum LoadingState {
  Unloaded,
  Loading,
  Loaded,
}

export function getTherapistName(t: TherapistEntity): string {
  return (t.title ? `${t.title} ` : "") + (t.name ?? "");
}

export function getPrintableAddress(therapist: TherapistEntity): string {
  return `${therapist.title ?? ""} ${therapist.name ?? ""}\n${therapist.street ?? ""} ${therapist.house_number ?? ""}\n${therapist.zip ?? ""} ${therapist.city ?? ""}`.trim();
}

export function getAddressWithoutName(therapist: TherapistEntity): string {
  return `${therapist.street ?? ""} ${therapist.house_number ?? ""} ${therapist.zip ?? ""} ${therapist.city ?? ""}`.trim();
}

export function recordMap<K extends string | number | symbol, V0, V1>(
  r: Record<K, V0>,
  f: (key: K, value: V0) => V1,
): Record<K, V1> {
  return Object.fromEntries(Object.entries(r).map(([k, v]) => [k, f(k, v)]));
}

export function recordFilter<K extends string | number | symbol, V>(
  r: Record<K, V>,
  pred: (key: K, value: V) => boolean,
): Record<K, V> {
  return Object.fromEntries(Object.entries(r).filter(([k, v]) => pred(k, v)));
}

// Can be used for debugging expressions
export function logExpr<V>(v: V): V {
  console.log(v);
  return v;
}

/**
 * partitions an array based on a filter function
 * @param a the array
 * @param pred the predicate function
 * @returns object with `right` and `wrong` keys
 */
export function partitionArray<t>(a: t[], pred: (x: t) => boolean) {
  const result: { right: t[]; wrong: t[] } = { right: [], wrong: [] };
  a.forEach((el) => (pred(el) ? result.right.push(el) : result.wrong.push(el)));
  return result;
}

export function timeToString(hour: number, minute: number) {
  return `${hour.toString().padStart(2, "0")}:${minute.toString().padStart(2, "0")}`;
}

/**
 * returns the timespan of a timeslot as a "hh:mm - hh:mm" string
 */
export function timeslotToString(ts: TimeslotEntity) {
  return `${timeToString(ts.start_hour, ts.start_minute)} - ${timeToString(ts.end_hour, ts.end_minute)}`;
}

export function getDayOfWeekAsString(dow: number) {
  return [
    "sunday",
    "monday",
    "tuesday",
    "wednesday",
    "thursday",
    "friday",
    "saturday",
  ][dow];
}
