/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import { ScrollView, TextInput } from "react-native";
import { useState } from "react";
import { Button, useTheme, Text, Portal, Dialog } from "react-native-paper";
import { SelectList } from "react-native-dropdown-select-list";
import { CallResult, CallResultLabels } from "@/constants/Protocol";
import Ionicons from "@expo/vector-icons/Ionicons";
import TherapistEntity from "@/data/entity/TherapistEntity";
import { createProtocolEntry, getTherapistsDS } from "@/utils/storage";
import TherapistPicker from "@/components/TherapistPicker";

export interface ProtocolFormProps {
  visible: boolean;
  setVisible: (b: boolean) => void;
  target?: TherapistEntity;
  onSave: (t: TherapistEntity) => void;
}

export default function ProtocolForm(props: ProtocolFormProps) {
  const theme = useTheme();
  const TherapistsDS = getTherapistsDS();
  const [resultValue, setResultValue] = useState(CallResult.SUCCESS);
  const [remark, setRemark] = useState<string>("");
  const [therapist, setTherapist] = useState(props.target);

  return (
    <Portal>
      <Dialog
        visible={props.visible}
        dismissable={true}
        dismissableBackButton={true}
        onDismiss={() => {
          props.setVisible(false);
        }}
      >
        <Dialog.Title>Neuer Protokolleintrag</Dialog.Title>
        <Dialog.ScrollArea>
          <ScrollView>
            {props.target == undefined && (
              <TherapistPicker
                selected={therapist}
                setSelected={setTherapist}
              />
            )}
            <Text>Ergebnis:</Text>
            <SelectList
              defaultOption={CallResultLabels[0]}
              search={false}
              boxStyles={{
                borderColor: theme.colors.background,
                backgroundColor: theme.colors.elevation.level2,
              }}
              dropdownStyles={{
                borderColor: theme.colors.background,
                backgroundColor: theme.colors.elevation.level2,
              }}
              inputStyles={{ color: theme.colors.onBackground }}
              dropdownTextStyles={{ color: theme.colors.onBackground }}
              arrowicon={
                <Ionicons
                  name="chevron-down"
                  color={theme.colors.primary}
                  size={15}
                ></Ionicons>
              }
              setSelected={setResultValue}
              data={CallResultLabels}
              save="key"
            />
            <Text>Bemerkung:</Text>
            <TextInput
              multiline={true}
              style={{
                color: theme.colors.onBackground,
                backgroundColor: theme.colors.elevation.level2,
                borderColor: theme.colors.elevation.level3,
                marginBottom: 10,
              }}
              placeholderTextColor={theme.colors.onBackground}
              value={remark}
              onChangeText={setRemark}
            />
          </ScrollView>
        </Dialog.ScrollArea>
        <Dialog.Actions>
          <Button
            disabled={!therapist}
            mode="contained"
            onPress={() => {
              if (therapist)
                createProtocolEntry(
                  TherapistsDS,
                  resultValue,
                  remark,
                  therapist,
                ).then(() => {
                  setResultValue(CallResult.SUCCESS);
                  setRemark("");
                  props.setVisible(false);
                  props.onSave(therapist);
                });
            }}
          >
            Speichern
          </Button>
        </Dialog.Actions>
      </Dialog>
    </Portal>
  );
}
