/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import { useEffect, useState } from "react";
import { Checkbox, Text } from "react-native-paper";
import {
  defaultSettings,
  loadSettings,
  Settings,
  storeSettings,
} from "@/utils/settings";
import { LoadingState } from "@/utils/misc";
import { Button, Platform, View } from "react-native";
import {
  backupDatabase,
  getTherapistsDS,
  restoreDatabase,
} from "@/utils/storage";
import NotificationPermissionButton from "@/components/NotificationPermissionButton";
import AlarmPermissionButton from "@/components/AlarmPermissionButton";

export default function SettingsPage() {
  const TherapistsDS = getTherapistsDS();

  const [loaded, setLoaded] = useState<LoadingState>(LoadingState.Unloaded);
  const [settings, setSettings] = useState<Settings>(defaultSettings);
  const [notificationPermission, setNotificationPermission] = useState(false);
  const [alarmPermission, setAlarmPermission] = useState(false);

  useEffect(() => {
    (async () => {
      if (loaded == LoadingState.Unloaded) {
        setLoaded(LoadingState.Loading);
        const s = await loadSettings();
        setSettings(s);
        setLoaded(LoadingState.Loaded);
      } else if (loaded == LoadingState.Loaded) {
        await storeSettings(settings);
      }
    })();
  }, [settings]);

  return (
    <View>
      <Checkbox.Item
        label={"Wilkommensseite beim Start der App anzeigen"}
        status={settings.showWelcome ? "checked" : "unchecked"}
        onPress={() => {
          setSettings({ showWelcome: !settings.showWelcome });
        }}
      />
      <NotificationPermissionButton
        notificationPermission={notificationPermission}
        setNotificationPermission={setNotificationPermission}
      />
      {Platform.OS === "android" && (
        <AlarmPermissionButton
          alarmPermission={alarmPermission}
          setAlarmPermission={setAlarmPermission}
        />
      )}
      <Button
        title={"Datenbank sichern"}
        onPress={() => {
          backupDatabase(TherapistsDS);
        }}
      />
      <Button
        title={"Datenbank wiederherstellen"}
        onPress={() => {
          restoreDatabase(TherapistsDS);
        }}
      />
      <Text>
        Bei Fragen, Problemen und Feedback: finn+therapiefinder@landweber.xyz
      </Text>
    </View>
  );
}
