/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import EditField from "@/components/EditField";
import TimeslotEntity from "@/data/entity/TimeslotEntity";
import TimeslotPicker from "@/components/TimeslotPicker";
import { useNavigation, useLocalSearchParams } from "expo-router";
import { useEffect, useState } from "react";
import { StyleSheet, ScrollView, View, Platform } from "react-native";
import {
  getTherapistsDS,
  loadTherapist,
  storeTherapist,
  deleteTherapist,
  createTherapist,
} from "@/utils/storage";
import { getTherapistName, LoadingState } from "@/utils/misc";
import TherapistEntity from "@/data/entity/TherapistEntity";
import { router } from "expo-router";
import { Checkbox, useTheme, Button, Divider } from "react-native-paper";

import { useHeaderHeight } from "@react-navigation/elements";

export default function EditTherapist() {
  const navigation = useNavigation();
  const TherapistsDS = getTherapistsDS();
  const { id } = useLocalSearchParams<{ id?: string | undefined }>();
  const theme = useTheme();
  const headerHeight = Platform.OS === "android" ? 0 : useHeaderHeight();

  const [therapist, setTherapist] = useState<TherapistEntity | undefined>(
    undefined,
  );
  const [originalTherapist, setOriginalTherapist] = useState<
    TherapistEntity | undefined
  >(undefined);
  const [loaded, setLoaded] = useState<LoadingState>(LoadingState.Unloaded);
  const [showTimeslots, setShowTimeslots] = useState(true);

  // boilerplate to update therapist, to be used with onChangeText
  const updateTherapist = (key: string) => (value: unknown) =>
    (async () => {
      //TODO: avoid duplicate `setTherapist`
      if (therapist === undefined) {
        //TODO: creation should return therapist *with relations* to avoid two db accesses
        //TODO: handle null case
        const newlyCreatedTherapist: TherapistEntity | null =
          await createTherapist(TherapistsDS);
        //TODO: handle null case
        const newlyCreatedTherapistProperlyLoaded: TherapistEntity | null =
          await loadTherapist(TherapistsDS, newlyCreatedTherapist.id);
        setTherapist({ ...newlyCreatedTherapistProperlyLoaded, [key]: value });
      } else setTherapist({ ...therapist, [key]: value });
    })();

  const ifTherapistDefined = <T,>(
    f: (t: TherapistEntity) => T,
  ): T | undefined => (therapist === undefined ? undefined : f(therapist));

  /* The model of this function is as follows:
     `loaded` is initialized with `Unloaded`, so on opening the screen it will first check if an id was passed and in that case load the corresponding `TherapistEntity` from the db. In either case it will then set `loaded` to `Loaded`, so that any further changes will not trigger reloading from the database.
     Any further calls of the function (whenever `therapist` changes) will consecutively write `therapist` to the db.
   */
  useEffect(() => {
    (async () => {
      if (loaded == LoadingState.Unloaded) {
        setLoaded(LoadingState.Loading);
        if (id !== undefined) {
          const t = await loadTherapist(TherapistsDS, Number(id));
          if (t != null) {
            setTherapist(t);
            setOriginalTherapist(t);
          } else {
            console.log("error while loading therapist with id", id);
          }
        }
        setLoaded(LoadingState.Loaded);
      } else if (loaded == LoadingState.Loaded && therapist !== undefined) {
        await storeTherapist(TherapistsDS, therapist);
      }
      if (therapist !== undefined)
        navigation.setOptions({ title: getTherapistName(therapist) });
    })();
  }, [therapist]);

  return (
    <ScrollView
      style={{ ...styles.container, paddingTop: headerHeight }}
      contentContainerStyle={styles.contentContainer}
    >
      {__DEV__ ? (
        <EditField
          title="Database ID"
          icon="rocket"
          value={ifTherapistDefined((t: TherapistEntity) => t.id.toString())}
          editable={false}
        />
      ) : undefined}

      <EditField
        icon="information"
        title="Titel"
        value={ifTherapistDefined((t: TherapistEntity) => t.title)}
        onChangeText={updateTherapist("title")}
      />
      <EditField
        icon="account"
        title="Name"
        value={ifTherapistDefined((t: TherapistEntity) => t.name)}
        onChangeText={updateTherapist("name")}
      />
      <EditField
        icon="phone"
        title="Telefon-Nr."
        value={ifTherapistDefined((t: TherapistEntity) => t.phone)}
        inputMode="tel"
        onChangeText={updateTherapist("phone")}
      />
      <EditField
        icon="at"
        title="E-Mail"
        value={ifTherapistDefined((t: TherapistEntity) => t.email)}
        inputMode="email"
        onChangeText={updateTherapist("email")}
      />
      <EditField
        icon="web"
        title="Website"
        value={ifTherapistDefined((t: TherapistEntity) => t.website)}
        onChangeText={updateTherapist("website")}
      />
      <EditField
        icon="map"
        title="Straße"
        value={ifTherapistDefined((t: TherapistEntity) => t.street)}
        onChangeText={updateTherapist("street")}
      />
      <EditField
        icon="home-group"
        title="Haus-Nr."
        value={ifTherapistDefined((t: TherapistEntity) => t.house_number)}
        onChangeText={updateTherapist("house_number")}
      />
      <EditField
        icon="mail"
        title="PLZ"
        value={ifTherapistDefined((t: TherapistEntity) => t.zip)}
        onChangeText={updateTherapist("zip")}
      />
      <EditField
        icon="city"
        title="Stadt"
        value={ifTherapistDefined((t: TherapistEntity) => t.city)}
        onChangeText={updateTherapist("city")}
      />
      <EditField
        icon="compass"
        title="Staat"
        value={ifTherapistDefined((t: TherapistEntity) => t.country)}
        onChangeText={updateTherapist("country")}
      />

      {ifTherapistDefined((t: TherapistEntity) => (
        <>
          {t.timeslots.length > 0 && (
            <View style={{ width: "100%" }}>
              <Checkbox.Item
                status={showTimeslots ? "checked" : "unchecked"}
                label={"Sprechzeiten anzeigen"}
                onPress={(_) => {
                  setShowTimeslots(!showTimeslots);
                }}
              />
            </View>
          )}
          {showTimeslots &&
            t.timeslots.map((item) => (
              <TimeslotPicker
                key={item.id}
                selectedTimeslot={item}
                setSelectedTimeslot={(newValue: TimeslotEntity) => {
                  updateTherapist("timeslots")(
                    t.timeslots.map((value) =>
                      value.id == item.id ? newValue : value,
                    ),
                  );
                }}
                deleteTimeslot={() => {
                  updateTherapist("timeslots")(
                    t.timeslots.filter(
                      (ts: TimeslotEntity) => ts.id != item.id,
                    ),
                  );
                }}
              />
            ))}
        </>
      ))}
      <View style={styles.bottomControls}>
        <Button
          onPress={() => {
            TherapistsDS.manager
              .save(TimeslotEntity, {
                day_of_week: "monday",
                start_hour: 0,
                start_minute: 0,
                end_hour: 0,
                end_minute: 0,
                therapist: therapist, //TODO: what if therapist undefined?
              })
              .then((newTsWithId) => {
                setShowTimeslots(true);
                updateTherapist("timeslots")([
                  ...(therapist === undefined ? [] : therapist.timeslots),
                  newTsWithId,
                ]);
              });
          }}
        >
          Erreichbarkeit hinzufügen
        </Button>

        <Checkbox.Item
          label={"In Anrufliste anzeigen"}
          status={
            (ifTherapistDefined((t: TherapistEntity) => t.enabled) ?? false)
              ? "checked"
              : "unchecked"
          }
          onPress={() => {
            updateTherapist("enabled")(!therapist?.enabled);
          }}
        />

        <Divider style={{ marginBottom: 10 }} />

        <Button
          onPress={() => {
            if (therapist !== undefined && originalTherapist === undefined)
              deleteTherapist(TherapistsDS, therapist.id).then((_) => {
                router.back();
              });
            else if (originalTherapist !== undefined) {
              storeTherapist(TherapistsDS, originalTherapist).then(router.back);
            }
          }}
          style={{ marginBottom: 10 }}
          buttonColor={theme.colors.errorContainer}
          textColor={theme.colors.onErrorContainer}
          icon="undo"
        >
          Abbrechen
        </Button>
        <Button
          onPress={() => {
            if (therapist !== undefined)
              deleteTherapist(TherapistsDS, therapist.id).then((_) => {
                router.back();
              });
            else router.back();
          }}
          buttonColor={theme.colors.errorContainer}
          textColor={theme.colors.onErrorContainer}
          icon="delete"
        >
          Löschen
        </Button>
      </View>
    </ScrollView>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, paddingVertical: 20 },
  contentContainer: { justifyContent: "center", alignItems: "center" },
  bottomControls: { flex: 1, marginBottom: 50, width: "90%" },
});
