/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import ProtocolListEntry from "@/components/ProtocolListEntry";
import ProtocolEntity from "@/data/entity/ProtocolEntity";

import { getTherapistsDS, loadProtocolEntries } from "@/utils/storage";

import { useFocusEffect } from "expo-router";
import React, { useCallback, useState } from "react";
import { FlatList, StyleSheet, View } from "react-native";
import { FAB, Portal, Text } from "react-native-paper";

import exportProtocolAsPDF from "@/utils/protocolExport";
import ProtocolForm from "@/components/ProtocolForm";

export default function Tab() {
  const TherapistsDS = getTherapistsDS();
  const [protocolEntries, setProtocolEntries] = useState<ProtocolEntity[]>([]);
  const [speedDialOpen, setSpeedDialOpen] = useState(false);
  const [showNewProtocol, setShowNewProtocol] = useState(false);

  const reloadProtocols = () => {
    loadProtocolEntries(TherapistsDS).then(setProtocolEntries);
  };

  useFocusEffect(
    useCallback(() => {
      reloadProtocols();
    }, []),
  );

  return (
    <Portal.Host>
      <View style={styles.container}>
        {protocolEntries.length > 0 ? (
          <>
            <FlatList
              style={{ width: "100%" }}
              data={protocolEntries}
              renderItem={(entry) => (
                <ProtocolListEntry
                  key={entry.item.id}
                  entry={entry.item}
                  onChange={() => {
                    reloadProtocols();
                  }}
                />
              )}
            />
          </>
        ) : (
          <>
            <Text style={{ textAlign: "center", margin: "auto" }}>
              Du hast noch keine Einträge im Protokoll.
            </Text>
          </>
        )}

        <Portal>
          <FAB.Group
            style={{ bottom: -40 }}
            open={speedDialOpen}
            visible
            icon={"dots-horizontal"}
            actions={[
              {
                icon: "plus",
                label: "Neuen Eintrag anlegen",
                onPress: () => {
                  setSpeedDialOpen(false);
                  setShowNewProtocol(true);
                },
              },
              {
                icon: "share",
                label: "Daten als PDF exportieren",
                onPress: () => {
                  exportProtocolAsPDF(TherapistsDS);
                  setSpeedDialOpen(false);
                },
              },
            ]}
            onStateChange={({ open }) => {
              setSpeedDialOpen(open);
            }}
          />
        </Portal>
        <ProtocolForm
          visible={showNewProtocol}
          setVisible={setShowNewProtocol}
          onSave={(_) => {
            reloadProtocols();
          }}
        />
      </View>
    </Portal.Host>
  );
}

const styles = StyleSheet.create({ container: { flex: 1 } });
