/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TherapistListEntry from "@/components/TherapistListEntry";
import { useRouter, useFocusEffect, useNavigation } from "expo-router";
import { View, StyleSheet, FlatList } from "react-native";
import {
  getTherapistsDS,
  loadTherapists,
  deleteTherapist,
} from "@/utils/storage";
import { getTherapistName } from "@/utils/misc";
import TherapistEntity from "@/data/entity/TherapistEntity";
import { useCallback, useEffect, useState } from "react";
import { Button, Dialog, Text, FAB, Portal } from "react-native-paper";
import Ionicons from "@expo/vector-icons/Ionicons";

export default function Tab() {
  const router = useRouter();
  const navigation = useNavigation();
  const TherapistsDS = getTherapistsDS();
  const [therapists, setTherapists] = useState<TherapistEntity[]>([]);
  const [speedDialOpen, setSpeedDialOpen] = useState(false);
  const [listRefreshing, setListRefreshing] = useState(false);
  const [deleteCandidate, setDeleteCandidate] = useState<number | null>(null);

  useFocusEffect(
    useCallback(() => {
      loadTherapists(TherapistsDS).then(setTherapists);
    }, []),
  );

  useEffect(() => {
    navigation.setOptions({
      headerRight: () => (
        <Button
          onPress={() => {
            router.push({ pathname: "/list/settings" });
          }}
        >
          <Ionicons name="settings-outline" size={24} />
        </Button>
      ),
    });
  }, []);

  const openEditPage = (id: number | undefined) => {
    router.push({
      pathname: `/list/edit`,
      params: id !== undefined ? { id: id } : {},
    });
  };

  const refreshTherapists = () => {
    setListRefreshing(true);
    loadTherapists(TherapistsDS).then((result) => {
      setTherapists(result);
      setListRefreshing(false);
    });
  };

  return (
    <Portal.Host>
      <View style={styles.listContainer}>
        {therapists.length == 0 ? (
          <>
            <View style={styles.backgroundContainer}>
              <Text style={{ textAlign: "center" }}>
                Aktuell keine Einträge
                {"\n\n"}
                Füge welche über das '+' hinzu!
              </Text>
            </View>
          </>
        ) : (
          <></>
        )}
        <FlatList
          onRefresh={refreshTherapists}
          refreshing={listRefreshing}
          data={therapists}
          renderItem={(item) => (
            <TherapistListEntry
              key={item.item.id}
              name={getTherapistName(item.item)}
              onPress={() => {
                openEditPage(item.item.id);
              }}
            />
          )}
        />
        <Portal>
          <FAB.Group
            style={{ bottom: -40 }}
            open={speedDialOpen}
            visible
            icon={"plus"}
            actions={[
              {
                icon: "plus",
                label: "Eintrag anlegen",
                onPress: () => {
                  setSpeedDialOpen(false);
                  openEditPage(undefined);
                },
              },
            ]}
            onStateChange={({ open }) => {
              setSpeedDialOpen(open);
            }}
          />
        </Portal>
        <Portal>
          <Dialog
            visible={deleteCandidate !== null}
            onDismiss={() => {
              setDeleteCandidate(null);
            }}
          >
            <Dialog.Title>Löschen</Dialog.Title>
            <Dialog.Content>
              <Text variant="bodyMedium">
                Möchtest du diesen Eintrag wirklich löschen?
              </Text>
            </Dialog.Content>
            <Dialog.Actions>
              <Button
                onPress={() => {
                  if (deleteCandidate !== null) {
                    deleteTherapist(TherapistsDS, deleteCandidate).then((_) => {
                      setDeleteCandidate(null);
                      refreshTherapists();
                    });
                  }
                }}
              >
                LÖSCHEN
              </Button>
              <Button
                onPress={() => {
                  setDeleteCandidate(null);
                }}
              >
                ABBRECHEN
              </Button>
            </Dialog.Actions>
          </Dialog>
        </Portal>
      </View>
    </Portal.Host>
  );
}

const styles = StyleSheet.create({
  container: {},
  listContainer: { flexGrow: 1 },
  backgroundContainer: {
    flex: 1,
    flexGrow: 1,
    position: "absolute",
    width: "100%",
    height: "100%",
    justifyContent: "center",
    alignItems: "center",
    padding: 30,
  },
});
